(function (global, factory) {
    typeof exports === 'object' && typeof module !== 'undefined' ? module.exports = factory() :
    typeof define === 'function' && define.amd ? define(factory) : 
	typeof define === 'function' && (define.cmd || define.hjs) ? define(function(require,exports,module){module.exports = factory()}) :
    (global.PixiApngAndGif = factory());
}(this, (function () { 'use strict';

    var $getExeName = (function (filePath) {
        var aList = filePath.split('.');
        return aList[aList.length - 1];
    });

    // (c) Dean McNamee <dean@gmail.com>, 2013.
    function GifReader(buf) {
        var p = 0;
        // - Header (GIF87a or GIF89a).
        if (buf[p++] !== 0x47 || buf[p++] !== 0x49 || buf[p++] !== 0x46 ||
            buf[p++] !== 0x38 || (buf[p++] + 1 & 0xfd) !== 0x38 || buf[p++] !== 0x61) {
            throw new Error("Invalid GIF 87a/89a header.");
        }
        // - Logical Screen Descriptor.
        var width = buf[p++] | buf[p++] << 8;
        var height = buf[p++] | buf[p++] << 8;
        var pf0 = buf[p++]; // <Packed Fields>.
        var global_palette_flag = pf0 >> 7;
        var num_global_colors_pow2 = pf0 & 0x7;
        var num_global_colors = 1 << (num_global_colors_pow2 + 1);
        var background = buf[p++];
        buf[p++]; // Pixel aspect ratio (unused?).
        var global_palette_offset = null;
        var global_palette_size = null;
        if (global_palette_flag) {
            global_palette_offset = p;
            global_palette_size = num_global_colors;
            p += num_global_colors * 3; // Seek past palette.
        }
        var no_eof = true;
        var frames = [];
        var delay = 0;
        var transparent_index = null;
        var disposal = 0; // 0 - No disposal specified.
        var loop_count = null;
        this.width = width;
        this.height = height;
        while (no_eof && p < buf.length) {
            switch (buf[p++]) {
                case 0x21: // Graphics Control Extension Block
                    switch (buf[p++]) {
                        case 0xff: // Application specific block
                            // Try if it's a Netscape block (with animation loop counter).
                            if (buf[p] !== 0x0b || // 21 FF already read, check block size.
                                // NETSCAPE2.0
                                buf[p + 1] == 0x4e && buf[p + 2] == 0x45 && buf[p + 3] == 0x54 &&
                                    buf[p + 4] == 0x53 && buf[p + 5] == 0x43 && buf[p + 6] == 0x41 &&
                                    buf[p + 7] == 0x50 && buf[p + 8] == 0x45 && buf[p + 9] == 0x32 &&
                                    buf[p + 10] == 0x2e && buf[p + 11] == 0x30 &&
                                    // Sub-block
                                    buf[p + 12] == 0x03 && buf[p + 13] == 0x01 && buf[p + 16] == 0) {
                                p += 14;
                                loop_count = buf[p++] | buf[p++] << 8;
                                p++; // Skip terminator.
                            }
                            else { // We don't know what it is, just try to get past it.
                                p += 12;
                                while (true) { // Seek through subblocks.
                                    var block_size = buf[p++];
                                    // Bad block size (ex: undefined from an out of bounds read).
                                    if (!(block_size >= 0))
                                        throw Error("Invalid block size");
                                    if (block_size === 0)
                                        break; // 0 size is terminator
                                    p += block_size;
                                }
                            }
                            break;
                        case 0xf9: // Graphics Control Extension
                            if (buf[p++] !== 0x4 || buf[p + 4] !== 0)
                                throw new Error("Invalid graphics extension block.");
                            var pf1 = buf[p++];
                            delay = buf[p++] | buf[p++] << 8;
                            transparent_index = buf[p++];
                            if ((pf1 & 1) === 0)
                                transparent_index = null;
                            disposal = pf1 >> 2 & 0x7;
                            p++; // Skip terminator.
                            break;
                        case 0xfe: // Comment Extension.
                            while (true) { // Seek through subblocks.
                                var block_size = buf[p++];
                                // Bad block size (ex: undefined from an out of bounds read).
                                if (!(block_size >= 0))
                                    throw Error("Invalid block size");
                                if (block_size === 0)
                                    break; // 0 size is terminator
                                // console.log(buf.slice(p, p+block_size).toString('ascii'));
                                p += block_size;
                            }
                            break;
                        default:
                            throw new Error("Unknown graphic control label: 0x" + buf[p - 1].toString(16));
                    }
                    break;
                case 0x2c: // Image Descriptor.
                    var x = buf[p++] | buf[p++] << 8;
                    var y = buf[p++] | buf[p++] << 8;
                    var w = buf[p++] | buf[p++] << 8;
                    var h = buf[p++] | buf[p++] << 8;
                    var pf2 = buf[p++];
                    var local_palette_flag = pf2 >> 7;
                    var interlace_flag = pf2 >> 6 & 1;
                    var num_local_colors_pow2 = pf2 & 0x7;
                    var num_local_colors = 1 << (num_local_colors_pow2 + 1);
                    var palette_offset = global_palette_offset;
                    var palette_size = global_palette_size;
                    var has_local_palette = false;
                    if (local_palette_flag) {
                        var has_local_palette = true;
                        palette_offset = p; // Override with local palette.
                        palette_size = num_local_colors;
                        p += num_local_colors * 3; // Seek past palette.
                    }
                    var data_offset = p;
                    p++; // codesize
                    while (true) {
                        var block_size = buf[p++];
                        // Bad block size (ex: undefined from an out of bounds read).
                        if (!(block_size >= 0))
                            throw Error("Invalid block size");
                        if (block_size === 0)
                            break; // 0 size is terminator
                        p += block_size;
                    }
                    frames.push({
                        x: x,
                        y: y,
                        width: w,
                        height: h,
                        has_local_palette: has_local_palette,
                        palette_offset: palette_offset,
                        palette_size: palette_size,
                        data_offset: data_offset,
                        data_length: p - data_offset,
                        transparent_index: transparent_index,
                        interlaced: !!interlace_flag,
                        delay: delay,
                        disposal: disposal
                    });
                    break;
                case 0x3b: // Trailer Marker (end of file).
                    no_eof = false;
                    break;
                default:
                    throw new Error("Unknown gif block: 0x" + buf[p - 1].toString(16));
                    break;
            }
        }
        this.numFrames = function () {
            return frames.length;
        };
        this.loopCount = function () {
            return loop_count;
        };
        this.frameInfo = function (frame_num) {
            if (frame_num < 0 || frame_num >= frames.length)
                throw new Error("Frame index out of range.");
            return frames[frame_num];
        };
        this.decodeAndBlitFrameBGRA = function (frame_num, pixels) {
            var frame = this.frameInfo(frame_num);
            var num_pixels = frame.width * frame.height;
            var index_stream = new Uint8Array(num_pixels); // At most 8-bit indices.
            GifReaderLZWOutputIndexStream(buf, frame.data_offset, index_stream, num_pixels);
            var palette_offset = frame.palette_offset;
            // NOTE(deanm): It seems to be much faster to compare index to 256 than
            // to === null.  Not sure why, but CompareStub_EQ_STRICT shows up high in
            // the profile, not sure if it's related to using a Uint8Array.
            var trans = frame.transparent_index;
            if (trans === null)
                trans = 256;
            // We are possibly just blitting to a portion of the entire frame.
            // That is a subrect within the framerect, so the additional pixels
            // must be skipped over after we finished a scanline.
            var framewidth = frame.width;
            var framestride = width - framewidth;
            var xleft = framewidth; // Number of subrect pixels left in scanline.
            // Output indicies of the top left and bottom right corners of the subrect.
            var opbeg = ((frame.y * width) + frame.x) * 4;
            var opend = ((frame.y + frame.height) * width + frame.x) * 4;
            var op = opbeg;
            var scanstride = framestride * 4;
            // Use scanstride to skip past the rows when interlacing.  This is skipping
            // 7 rows for the first two passes, then 3 then 1.
            if (frame.interlaced === true) {
                scanstride += width * 4 * 7; // Pass 1.
            }
            var interlaceskip = 8; // Tracking the row interval in the current pass.
            for (var i = 0, il = index_stream.length; i < il; ++i) {
                var index = index_stream[i];
                if (xleft === 0) { // Beginning of new scan line
                    op += scanstride;
                    xleft = framewidth;
                    if (op >= opend) { // Catch the wrap to switch passes when interlacing.
                        scanstride = framestride * 4 + width * 4 * (interlaceskip - 1);
                        // interlaceskip / 2 * 4 is interlaceskip << 1.
                        op = opbeg + (framewidth + framestride) * (interlaceskip << 1);
                        interlaceskip >>= 1;
                    }
                }
                if (index === trans) {
                    op += 4;
                }
                else {
                    var r = buf[palette_offset + index * 3];
                    var g = buf[palette_offset + index * 3 + 1];
                    var b = buf[palette_offset + index * 3 + 2];
                    pixels[op++] = b;
                    pixels[op++] = g;
                    pixels[op++] = r;
                    pixels[op++] = 255;
                }
                --xleft;
            }
        };
        // I will go to copy and paste hell one day...
        this.decodeAndBlitFrameRGBA = function (frame_num, pixels) {
            var frame = this.frameInfo(frame_num);
            var num_pixels = frame.width * frame.height;
            var index_stream = new Uint8Array(num_pixels); // At most 8-bit indices.
            GifReaderLZWOutputIndexStream(buf, frame.data_offset, index_stream, num_pixels);
            var palette_offset = frame.palette_offset;
            // NOTE(deanm): It seems to be much faster to compare index to 256 than
            // to === null.  Not sure why, but CompareStub_EQ_STRICT shows up high in
            // the profile, not sure if it's related to using a Uint8Array.
            var trans = frame.transparent_index;
            if (trans === null)
                trans = 256;
            // We are possibly just blitting to a portion of the entire frame.
            // That is a subrect within the framerect, so the additional pixels
            // must be skipped over after we finished a scanline.
            var framewidth = frame.width;
            var framestride = width - framewidth;
            var xleft = framewidth; // Number of subrect pixels left in scanline.
            // Output indicies of the top left and bottom right corners of the subrect.
            var opbeg = ((frame.y * width) + frame.x) * 4;
            var opend = ((frame.y + frame.height) * width + frame.x) * 4;
            var op = opbeg;
            var scanstride = framestride * 4;
            // Use scanstride to skip past the rows when interlacing.  This is skipping
            // 7 rows for the first two passes, then 3 then 1.
            if (frame.interlaced === true) {
                scanstride += width * 4 * 7; // Pass 1.
            }
            var interlaceskip = 8; // Tracking the row interval in the current pass.
            for (var i = 0, il = index_stream.length; i < il; ++i) {
                var index = index_stream[i];
                if (xleft === 0) { // Beginning of new scan line
                    op += scanstride;
                    xleft = framewidth;
                    if (op >= opend) { // Catch the wrap to switch passes when interlacing.
                        scanstride = framestride * 4 + width * 4 * (interlaceskip - 1);
                        // interlaceskip / 2 * 4 is interlaceskip << 1.
                        op = opbeg + (framewidth + framestride) * (interlaceskip << 1);
                        interlaceskip >>= 1;
                    }
                }
                if (index === trans) {
                    op += 4;
                }
                else {
                    var r = buf[palette_offset + index * 3];
                    var g = buf[palette_offset + index * 3 + 1];
                    var b = buf[palette_offset + index * 3 + 2];
                    pixels[op++] = r;
                    pixels[op++] = g;
                    pixels[op++] = b;
                    pixels[op++] = 255;
                }
                --xleft;
            }
        };
    }
    function GifReaderLZWOutputIndexStream(code_stream, p, output, output_length) {
        var min_code_size = code_stream[p++];
        var clear_code = 1 << min_code_size;
        var eoi_code = clear_code + 1;
        var next_code = eoi_code + 1;
        var cur_code_size = min_code_size + 1; // Number of bits per code.
        // NOTE: This shares the same name as the encoder, but has a different
        // meaning here.  Here this masks each code coming from the code stream.
        var code_mask = (1 << cur_code_size) - 1;
        var cur_shift = 0;
        var cur = 0;
        var op = 0; // Output pointer.
        var subblock_size = code_stream[p++];
        // TODO(deanm): Would using a TypedArray be any faster?  At least it would
        // solve the fast mode / backing store uncertainty.
        // var code_table = Array(4096);
        var code_table = new Int32Array(4096); // Can be signed, we only use 20 bits.
        var prev_code = null; // Track code-1.
        while (true) {
            // Read up to two bytes, making sure we always 12-bits for max sized code.
            while (cur_shift < 16) {
                if (subblock_size === 0)
                    break; // No more data to be read.
                cur |= code_stream[p++] << cur_shift;
                cur_shift += 8;
                if (subblock_size === 1) { // Never let it get to 0 to hold logic above.
                    subblock_size = code_stream[p++]; // Next subblock.
                }
                else {
                    --subblock_size;
                }
            }
            // TODO(deanm): We should never really get here, we should have received
            // and EOI.
            if (cur_shift < cur_code_size)
                break;
            var code = cur & code_mask;
            cur >>= cur_code_size;
            cur_shift -= cur_code_size;
            // TODO(deanm): Maybe should check that the first code was a clear code,
            // at least this is what you're supposed to do.  But actually our encoder
            // now doesn't emit a clear code first anyway.
            if (code === clear_code) {
                // We don't actually have to clear the table.  This could be a good idea
                // for greater error checking, but we don't really do any anyway.  We
                // will just track it with next_code and overwrite old entries.
                next_code = eoi_code + 1;
                cur_code_size = min_code_size + 1;
                code_mask = (1 << cur_code_size) - 1;
                // Don't update prev_code ?
                prev_code = null;
                continue;
            }
            else if (code === eoi_code) {
                break;
            }
            // We have a similar situation as the decoder, where we want to store
            // variable length entries (code table entries), but we want to do in a
            // faster manner than an array of arrays.  The code below stores sort of a
            // linked list within the code table, and then "chases" through it to
            // construct the dictionary entries.  When a new entry is created, just the
            // last byte is stored, and the rest (prefix) of the entry is only
            // referenced by its table entry.  Then the code chases through the
            // prefixes until it reaches a single byte code.  We have to chase twice,
            // first to compute the length, and then to actually copy the data to the
            // output (backwards, since we know the length).  The alternative would be
            // storing something in an intermediate stack, but that doesn't make any
            // more sense.  I implemented an approach where it also stored the length
            // in the code table, although it's a bit tricky because you run out of
            // bits (12 + 12 + 8), but I didn't measure much improvements (the table
            // entries are generally not the long).  Even when I created benchmarks for
            // very long table entries the complexity did not seem worth it.
            // The code table stores the prefix entry in 12 bits and then the suffix
            // byte in 8 bits, so each entry is 20 bits.
            var chase_code = code < next_code ? code : prev_code;
            // Chase what we will output, either {CODE} or {CODE-1}.
            var chase_length = 0;
            var chase = chase_code;
            while (chase > clear_code) {
                chase = code_table[chase] >> 8;
                ++chase_length;
            }
            var k = chase;
            var op_end = op + chase_length + (chase_code !== code ? 1 : 0);
            if (op_end > output_length) {
                console.log("Warning, gif stream longer than expected.");
                return;
            }
            // Already have the first byte from the chase, might as well write it fast.
            output[op++] = k;
            op += chase_length;
            var b = op; // Track pointer, writing backwards.
            if (chase_code !== code) // The case of emitting {CODE-1} + k.
                output[op++] = k;
            chase = chase_code;
            while (chase_length--) {
                chase = code_table[chase];
                output[--b] = chase & 0xff; // Write backwards.
                chase >>= 8; // Pull down to the prefix code.
            }
            if (prev_code !== null && next_code < 4096) {
                code_table[next_code++] = prev_code << 8 | k;
                // TODO(deanm): Figure out this clearing vs code growth logic better.  I
                // have an feeling that it should just happen somewhere else, for now it
                // is awkward between when we grow past the max and then hit a clear code.
                // For now just check if we hit the max 12-bits (then a clear code should
                // follow, also of course encoded in 12-bits).
                if (next_code >= code_mask + 1 && cur_code_size < 12) {
                    ++cur_code_size;
                    code_mask = code_mask << 1 | 1;
                }
            }
            prev_code = code;
        }
        if (op !== output_length) {
            console.log("Warning, gif stream shorter than expected.");
        }
        return output;
    }

    function createCommonjsModule(fn, module) {
        return module = { exports: {} }, fn(module, module.exports), module.exports;
    }

    var common = createCommonjsModule(function (module, exports) {
        var TYPED_OK = (typeof Uint8Array !== 'undefined') &&
            (typeof Uint16Array !== 'undefined') &&
            (typeof Int32Array !== 'undefined');
        function _has(obj, key) {
            return Object.prototype.hasOwnProperty.call(obj, key);
        }
        exports.assign = function (obj /*from1, from2, from3, ...*/) {
            var sources = Array.prototype.slice.call(arguments, 1);
            while (sources.length) {
                var source = sources.shift();
                if (!source) {
                    continue;
                }
                if (typeof source !== 'object') {
                    throw new TypeError(source + 'must be non-object');
                }
                for (var p in source) {
                    if (_has(source, p)) {
                        obj[p] = source[p];
                    }
                }
            }
            return obj;
        };
        // reduce buffer size, avoiding mem copy
        exports.shrinkBuf = function (buf, size) {
            if (buf.length === size) {
                return buf;
            }
            if (buf.subarray) {
                return buf.subarray(0, size);
            }
            buf.length = size;
            return buf;
        };
        var fnTyped = {
            arraySet: function (dest, src, src_offs, len, dest_offs) {
                if (src.subarray && dest.subarray) {
                    dest.set(src.subarray(src_offs, src_offs + len), dest_offs);
                    return;
                }
                // Fallback to ordinary array
                for (var i = 0; i < len; i++) {
                    dest[dest_offs + i] = src[src_offs + i];
                }
            },
            // Join array of chunks to single array.
            flattenChunks: function (chunks) {
                var i, l, len, pos, chunk, result;
                // calculate data length
                len = 0;
                for (i = 0, l = chunks.length; i < l; i++) {
                    len += chunks[i].length;
                }
                // join chunks
                result = new Uint8Array(len);
                pos = 0;
                for (i = 0, l = chunks.length; i < l; i++) {
                    chunk = chunks[i];
                    result.set(chunk, pos);
                    pos += chunk.length;
                }
                return result;
            }
        };
        var fnUntyped = {
            arraySet: function (dest, src, src_offs, len, dest_offs) {
                for (var i = 0; i < len; i++) {
                    dest[dest_offs + i] = src[src_offs + i];
                }
            },
            // Join array of chunks to single array.
            flattenChunks: function (chunks) {
                return [].concat.apply([], chunks);
            }
        };
        // Enable/Disable typed arrays use, for testing
        //
        exports.setTyped = function (on) {
            if (on) {
                exports.Buf8 = Uint8Array;
                exports.Buf16 = Uint16Array;
                exports.Buf32 = Int32Array;
                exports.assign(exports, fnTyped);
            }
            else {
                exports.Buf8 = Array;
                exports.Buf16 = Array;
                exports.Buf32 = Array;
                exports.assign(exports, fnUntyped);
            }
        };
        exports.setTyped(TYPED_OK);
    });
    var common_1 = common.assign;
    var common_2 = common.shrinkBuf;
    var common_3 = common.setTyped;
    var common_4 = common.Buf8;
    var common_5 = common.Buf16;
    var common_6 = common.Buf32;

    var common$1 = /*#__PURE__*/Object.freeze({
        default: common,
        __moduleExports: common,
        assign: common_1,
        shrinkBuf: common_2,
        setTyped: common_3,
        Buf8: common_4,
        Buf16: common_5,
        Buf32: common_6
    });

    var utils = (common$1 && common) || common$1;

    // (C) 1995-2013 Jean-loup Gailly and Mark Adler
    // (C) 2014-2017 Vitaly Puzrin and Andrey Tupitsin
    //
    // This software is provided 'as-is', without any express or implied
    // warranty. In no event will the authors be held liable for any damages
    // arising from the use of this software.
    //
    // Permission is granted to anyone to use this software for any purpose,
    // including commercial applications, and to alter it and redistribute it
    // freely, subject to the following restrictions:
    //
    // 1. The origin of this software must not be misrepresented; you must not
    //   claim that you wrote the original software. If you use this software
    //   in a product, an acknowledgment in the product documentation would be
    //   appreciated but is not required.
    // 2. Altered source versions must be plainly marked as such, and must not be
    //   misrepresented as being the original software.
    // 3. This notice may not be removed or altered from any source distribution.
    /* eslint-disable space-unary-ops */
    /* Public constants ==========================================================*/
    /* ===========================================================================*/
    //var Z_FILTERED          = 1;
    //var Z_HUFFMAN_ONLY      = 2;
    //var Z_RLE               = 3;
    var Z_FIXED = 4;
    //var Z_DEFAULT_STRATEGY  = 0;
    /* Possible values of the data_type field (though see inflate()) */
    var Z_BINARY = 0;
    var Z_TEXT = 1;
    //var Z_ASCII             = 1; // = Z_TEXT
    var Z_UNKNOWN = 2;
    /*============================================================================*/
    function zero(buf) { var len = buf.length; while (--len >= 0) {
        buf[len] = 0;
    } }
    // From zutil.h
    var STORED_BLOCK = 0;
    var STATIC_TREES = 1;
    var DYN_TREES = 2;
    /* The three kinds of block type */
    var MIN_MATCH = 3;
    var MAX_MATCH = 258;
    /* The minimum and maximum match lengths */
    // From deflate.h
    /* ===========================================================================
     * Internal compression state.
     */
    var LENGTH_CODES = 29;
    /* number of length codes, not counting the special END_BLOCK code */
    var LITERALS = 256;
    /* number of literal bytes 0..255 */
    var L_CODES = LITERALS + 1 + LENGTH_CODES;
    /* number of Literal or Length codes, including the END_BLOCK code */
    var D_CODES = 30;
    /* number of distance codes */
    var BL_CODES = 19;
    /* number of codes used to transfer the bit lengths */
    var HEAP_SIZE = 2 * L_CODES + 1;
    /* maximum heap size */
    var MAX_BITS = 15;
    /* All codes must not exceed MAX_BITS bits */
    var Buf_size = 16;
    /* size of bit buffer in bi_buf */
    /* ===========================================================================
     * Constants
     */
    var MAX_BL_BITS = 7;
    /* Bit length codes must not exceed MAX_BL_BITS bits */
    var END_BLOCK = 256;
    /* end of block literal code */
    var REP_3_6 = 16;
    /* repeat previous bit length 3-6 times (2 bits of repeat count) */
    var REPZ_3_10 = 17;
    /* repeat a zero length 3-10 times  (3 bits of repeat count) */
    var REPZ_11_138 = 18;
    /* repeat a zero length 11-138 times  (7 bits of repeat count) */
    /* eslint-disable comma-spacing,array-bracket-spacing */
    var extra_lbits = /* extra bits for each length code */ [0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 2, 2, 2, 2, 3, 3, 3, 3, 4, 4, 4, 4, 5, 5, 5, 5, 0];
    var extra_dbits = /* extra bits for each distance code */ [0, 0, 0, 0, 1, 1, 2, 2, 3, 3, 4, 4, 5, 5, 6, 6, 7, 7, 8, 8, 9, 9, 10, 10, 11, 11, 12, 12, 13, 13];
    var extra_blbits = /* extra bits for each bit length code */ [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 3, 7];
    var bl_order = [16, 17, 18, 0, 8, 7, 9, 6, 10, 5, 11, 4, 12, 3, 13, 2, 14, 1, 15];
    /* eslint-enable comma-spacing,array-bracket-spacing */
    /* The lengths of the bit length codes are sent in order of decreasing
     * probability, to avoid transmitting the lengths for unused bit length codes.
     */
    /* ===========================================================================
     * Local data. These are initialized only once.
     */
    // We pre-fill arrays with 0 to avoid uninitialized gaps
    var DIST_CODE_LEN = 512; /* see definition of array dist_code below */
    // !!!! Use flat array instead of structure, Freq = i*2, Len = i*2+1
    var static_ltree = new Array((L_CODES + 2) * 2);
    zero(static_ltree);
    /* The static literal tree. Since the bit lengths are imposed, there is no
     * need for the L_CODES extra codes used during heap construction. However
     * The codes 286 and 287 are needed to build a canonical tree (see _tr_init
     * below).
     */
    var static_dtree = new Array(D_CODES * 2);
    zero(static_dtree);
    /* The static distance tree. (Actually a trivial tree since all codes use
     * 5 bits.)
     */
    var _dist_code = new Array(DIST_CODE_LEN);
    zero(_dist_code);
    /* Distance codes. The first 256 values correspond to the distances
     * 3 .. 258, the last 256 values correspond to the top 8 bits of
     * the 15 bit distances.
     */
    var _length_code = new Array(MAX_MATCH - MIN_MATCH + 1);
    zero(_length_code);
    /* length code for each normalized match length (0 == MIN_MATCH) */
    var base_length = new Array(LENGTH_CODES);
    zero(base_length);
    /* First normalized length for each code (0 = MIN_MATCH) */
    var base_dist = new Array(D_CODES);
    zero(base_dist);
    /* First normalized distance for each code (0 = distance of 1) */
    function StaticTreeDesc(static_tree, extra_bits, extra_base, elems, max_length) {
        this.static_tree = static_tree; /* static tree or NULL */
        this.extra_bits = extra_bits; /* extra bits for each code or NULL */
        this.extra_base = extra_base; /* base index for extra_bits */
        this.elems = elems; /* max number of elements in the tree */
        this.max_length = max_length; /* max bit length for the codes */
        // show if `static_tree` has data or dummy - needed for monomorphic objects
        this.has_stree = static_tree && static_tree.length;
    }
    var static_l_desc;
    var static_d_desc;
    var static_bl_desc;
    function TreeDesc(dyn_tree, stat_desc) {
        this.dyn_tree = dyn_tree; /* the dynamic tree */
        this.max_code = 0; /* largest code with non zero frequency */
        this.stat_desc = stat_desc; /* the corresponding static tree */
    }
    function d_code(dist) {
        return dist < 256 ? _dist_code[dist] : _dist_code[256 + (dist >>> 7)];
    }
    /* ===========================================================================
     * Output a short LSB first on the stream.
     * IN assertion: there is enough room in pendingBuf.
     */
    function put_short(s, w) {
        //    put_byte(s, (uch)((w) & 0xff));
        //    put_byte(s, (uch)((ush)(w) >> 8));
        s.pending_buf[s.pending++] = (w) & 0xff;
        s.pending_buf[s.pending++] = (w >>> 8) & 0xff;
    }
    /* ===========================================================================
     * Send a value on a given number of bits.
     * IN assertion: length <= 16 and value fits in length bits.
     */
    function send_bits(s, value, length) {
        if (s.bi_valid > (Buf_size - length)) {
            s.bi_buf |= (value << s.bi_valid) & 0xffff;
            put_short(s, s.bi_buf);
            s.bi_buf = value >> (Buf_size - s.bi_valid);
            s.bi_valid += length - Buf_size;
        }
        else {
            s.bi_buf |= (value << s.bi_valid) & 0xffff;
            s.bi_valid += length;
        }
    }
    function send_code(s, c, tree) {
        send_bits(s, tree[c * 2] /*.Code*/, tree[c * 2 + 1] /*.Len*/);
    }
    /* ===========================================================================
     * Reverse the first len bits of a code, using straightforward code (a faster
     * method would use a table)
     * IN assertion: 1 <= len <= 15
     */
    function bi_reverse(code, len) {
        var res = 0;
        do {
            res |= code & 1;
            code >>>= 1;
            res <<= 1;
        } while (--len > 0);
        return res >>> 1;
    }
    /* ===========================================================================
     * Flush the bit buffer, keeping at most 7 bits in it.
     */
    function bi_flush(s) {
        if (s.bi_valid === 16) {
            put_short(s, s.bi_buf);
            s.bi_buf = 0;
            s.bi_valid = 0;
        }
        else if (s.bi_valid >= 8) {
            s.pending_buf[s.pending++] = s.bi_buf & 0xff;
            s.bi_buf >>= 8;
            s.bi_valid -= 8;
        }
    }
    /* ===========================================================================
     * Compute the optimal bit lengths for a tree and update the total bit length
     * for the current block.
     * IN assertion: the fields freq and dad are set, heap[heap_max] and
     *    above are the tree nodes sorted by increasing frequency.
     * OUT assertions: the field len is set to the optimal bit length, the
     *     array bl_count contains the frequencies for each bit length.
     *     The length opt_len is updated; static_len is also updated if stree is
     *     not null.
     */
    function gen_bitlen(s, desc) {
        var tree = desc.dyn_tree;
        var max_code = desc.max_code;
        var stree = desc.stat_desc.static_tree;
        var has_stree = desc.stat_desc.has_stree;
        var extra = desc.stat_desc.extra_bits;
        var base = desc.stat_desc.extra_base;
        var max_length = desc.stat_desc.max_length;
        var h; /* heap index */
        var n, m; /* iterate over the tree elements */
        var bits; /* bit length */
        var xbits; /* extra bits */
        var f; /* frequency */
        var overflow = 0; /* number of elements with bit length too large */
        for (bits = 0; bits <= MAX_BITS; bits++) {
            s.bl_count[bits] = 0;
        }
        /* In a first pass, compute the optimal bit lengths (which may
         * overflow in the case of the bit length tree).
         */
        tree[s.heap[s.heap_max] * 2 + 1] /*.Len*/ = 0; /* root of the heap */
        for (h = s.heap_max + 1; h < HEAP_SIZE; h++) {
            n = s.heap[h];
            bits = tree[tree[n * 2 + 1] /*.Dad*/ * 2 + 1] /*.Len*/ + 1;
            if (bits > max_length) {
                bits = max_length;
                overflow++;
            }
            tree[n * 2 + 1] /*.Len*/ = bits;
            /* We overwrite tree[n].Dad which is no longer needed */
            if (n > max_code) {
                continue;
            } /* not a leaf node */
            s.bl_count[bits]++;
            xbits = 0;
            if (n >= base) {
                xbits = extra[n - base];
            }
            f = tree[n * 2] /*.Freq*/;
            s.opt_len += f * (bits + xbits);
            if (has_stree) {
                s.static_len += f * (stree[n * 2 + 1] /*.Len*/ + xbits);
            }
        }
        if (overflow === 0) {
            return;
        }
        // Trace((stderr,"\nbit length overflow\n"));
        /* This happens for example on obj2 and pic of the Calgary corpus */
        /* Find the first bit length which could increase: */
        do {
            bits = max_length - 1;
            while (s.bl_count[bits] === 0) {
                bits--;
            }
            s.bl_count[bits]--; /* move one leaf down the tree */
            s.bl_count[bits + 1] += 2; /* move one overflow item as its brother */
            s.bl_count[max_length]--;
            /* The brother of the overflow item also moves one step up,
             * but this does not affect bl_count[max_length]
             */
            overflow -= 2;
        } while (overflow > 0);
        /* Now recompute all bit lengths, scanning in increasing frequency.
         * h is still equal to HEAP_SIZE. (It is simpler to reconstruct all
         * lengths instead of fixing only the wrong ones. This idea is taken
         * from 'ar' written by Haruhiko Okumura.)
         */
        for (bits = max_length; bits !== 0; bits--) {
            n = s.bl_count[bits];
            while (n !== 0) {
                m = s.heap[--h];
                if (m > max_code) {
                    continue;
                }
                if (tree[m * 2 + 1] /*.Len*/ !== bits) {
                    // Trace((stderr,"code %d bits %d->%d\n", m, tree[m].Len, bits));
                    s.opt_len += (bits - tree[m * 2 + 1] /*.Len*/) * tree[m * 2] /*.Freq*/;
                    tree[m * 2 + 1] /*.Len*/ = bits;
                }
                n--;
            }
        }
    }
    /* ===========================================================================
     * Generate the codes for a given tree and bit counts (which need not be
     * optimal).
     * IN assertion: the array bl_count contains the bit length statistics for
     * the given tree and the field len is set for all tree elements.
     * OUT assertion: the field code is set for all tree elements of non
     *     zero code length.
     */
    function gen_codes(tree, max_code, bl_count) {
        var next_code = new Array(MAX_BITS + 1); /* next code value for each bit length */
        var code = 0; /* running code value */
        var bits; /* bit index */
        var n; /* code index */
        /* The distribution counts are first used to generate the code values
         * without bit reversal.
         */
        for (bits = 1; bits <= MAX_BITS; bits++) {
            next_code[bits] = code = (code + bl_count[bits - 1]) << 1;
        }
        /* Check that the bit counts in bl_count are consistent. The last code
         * must be all ones.
         */
        //Assert (code + bl_count[MAX_BITS]-1 == (1<<MAX_BITS)-1,
        //        "inconsistent bit counts");
        //Tracev((stderr,"\ngen_codes: max_code %d ", max_code));
        for (n = 0; n <= max_code; n++) {
            var len = tree[n * 2 + 1] /*.Len*/;
            if (len === 0) {
                continue;
            }
            /* Now reverse the bits */
            tree[n * 2] /*.Code*/ = bi_reverse(next_code[len]++, len);
            //Tracecv(tree != static_ltree, (stderr,"\nn %3d %c l %2d c %4x (%x) ",
            //     n, (isgraph(n) ? n : ' '), len, tree[n].Code, next_code[len]-1));
        }
    }
    /* ===========================================================================
     * Initialize the various 'constant' tables.
     */
    function tr_static_init() {
        var n; /* iterates over tree elements */
        var bits; /* bit counter */
        var length; /* length value */
        var code; /* code value */
        var dist; /* distance index */
        var bl_count = new Array(MAX_BITS + 1);
        /* number of codes at each bit length for an optimal tree */
        // do check in _tr_init()
        //if (static_init_done) return;
        /* For some embedded targets, global variables are not initialized: */
        /*#ifdef NO_INIT_GLOBAL_POINTERS
          static_l_desc.static_tree = static_ltree;
          static_l_desc.extra_bits = extra_lbits;
          static_d_desc.static_tree = static_dtree;
          static_d_desc.extra_bits = extra_dbits;
          static_bl_desc.extra_bits = extra_blbits;
        #endif*/
        /* Initialize the mapping length (0..255) -> length code (0..28) */
        length = 0;
        for (code = 0; code < LENGTH_CODES - 1; code++) {
            base_length[code] = length;
            for (n = 0; n < (1 << extra_lbits[code]); n++) {
                _length_code[length++] = code;
            }
        }
        //Assert (length == 256, "tr_static_init: length != 256");
        /* Note that the length 255 (match length 258) can be represented
         * in two different ways: code 284 + 5 bits or code 285, so we
         * overwrite length_code[255] to use the best encoding:
         */
        _length_code[length - 1] = code;
        /* Initialize the mapping dist (0..32K) -> dist code (0..29) */
        dist = 0;
        for (code = 0; code < 16; code++) {
            base_dist[code] = dist;
            for (n = 0; n < (1 << extra_dbits[code]); n++) {
                _dist_code[dist++] = code;
            }
        }
        //Assert (dist == 256, "tr_static_init: dist != 256");
        dist >>= 7; /* from now on, all distances are divided by 128 */
        for (; code < D_CODES; code++) {
            base_dist[code] = dist << 7;
            for (n = 0; n < (1 << (extra_dbits[code] - 7)); n++) {
                _dist_code[256 + dist++] = code;
            }
        }
        //Assert (dist == 256, "tr_static_init: 256+dist != 512");
        /* Construct the codes of the static literal tree */
        for (bits = 0; bits <= MAX_BITS; bits++) {
            bl_count[bits] = 0;
        }
        n = 0;
        while (n <= 143) {
            static_ltree[n * 2 + 1] /*.Len*/ = 8;
            n++;
            bl_count[8]++;
        }
        while (n <= 255) {
            static_ltree[n * 2 + 1] /*.Len*/ = 9;
            n++;
            bl_count[9]++;
        }
        while (n <= 279) {
            static_ltree[n * 2 + 1] /*.Len*/ = 7;
            n++;
            bl_count[7]++;
        }
        while (n <= 287) {
            static_ltree[n * 2 + 1] /*.Len*/ = 8;
            n++;
            bl_count[8]++;
        }
        /* Codes 286 and 287 do not exist, but we must include them in the
         * tree construction to get a canonical Huffman tree (longest code
         * all ones)
         */
        gen_codes(static_ltree, L_CODES + 1, bl_count);
        /* The static distance tree is trivial: */
        for (n = 0; n < D_CODES; n++) {
            static_dtree[n * 2 + 1] /*.Len*/ = 5;
            static_dtree[n * 2] /*.Code*/ = bi_reverse(n, 5);
        }
        // Now data ready and we can init static trees
        static_l_desc = new StaticTreeDesc(static_ltree, extra_lbits, LITERALS + 1, L_CODES, MAX_BITS);
        static_d_desc = new StaticTreeDesc(static_dtree, extra_dbits, 0, D_CODES, MAX_BITS);
        static_bl_desc = new StaticTreeDesc(new Array(0), extra_blbits, 0, BL_CODES, MAX_BL_BITS);
        //static_init_done = true;
    }
    /* ===========================================================================
     * Initialize a new block.
     */
    function init_block(s) {
        var n; /* iterates over tree elements */
        /* Initialize the trees. */
        for (n = 0; n < L_CODES; n++) {
            s.dyn_ltree[n * 2] /*.Freq*/ = 0;
        }
        for (n = 0; n < D_CODES; n++) {
            s.dyn_dtree[n * 2] /*.Freq*/ = 0;
        }
        for (n = 0; n < BL_CODES; n++) {
            s.bl_tree[n * 2] /*.Freq*/ = 0;
        }
        s.dyn_ltree[END_BLOCK * 2] /*.Freq*/ = 1;
        s.opt_len = s.static_len = 0;
        s.last_lit = s.matches = 0;
    }
    /* ===========================================================================
     * Flush the bit buffer and align the output on a byte boundary
     */
    function bi_windup(s) {
        if (s.bi_valid > 8) {
            put_short(s, s.bi_buf);
        }
        else if (s.bi_valid > 0) {
            //put_byte(s, (Byte)s->bi_buf);
            s.pending_buf[s.pending++] = s.bi_buf;
        }
        s.bi_buf = 0;
        s.bi_valid = 0;
    }
    /* ===========================================================================
     * Copy a stored block, storing first the length and its
     * one's complement if requested.
     */
    function copy_block(s, buf, len, header) {
        bi_windup(s); /* align on byte boundary */
        if (header) {
            put_short(s, len);
            put_short(s, ~len);
        }
        //  while (len--) {
        //    put_byte(s, *buf++);
        //  }
        utils.arraySet(s.pending_buf, s.window, buf, len, s.pending);
        s.pending += len;
    }
    /* ===========================================================================
     * Compares to subtrees, using the tree depth as tie breaker when
     * the subtrees have equal frequency. This minimizes the worst case length.
     */
    function smaller(tree, n, m, depth) {
        var _n2 = n * 2;
        var _m2 = m * 2;
        return (tree[_n2] /*.Freq*/ < tree[_m2] /*.Freq*/ ||
            (tree[_n2] /*.Freq*/ === tree[_m2] /*.Freq*/ && depth[n] <= depth[m]));
    }
    /* ===========================================================================
     * Restore the heap property by moving down the tree starting at node k,
     * exchanging a node with the smallest of its two sons if necessary, stopping
     * when the heap property is re-established (each father smaller than its
     * two sons).
     */
    function pqdownheap(s, tree, k) {
        var v = s.heap[k];
        var j = k << 1; /* left son of k */
        while (j <= s.heap_len) {
            /* Set j to the smallest of the two sons: */
            if (j < s.heap_len &&
                smaller(tree, s.heap[j + 1], s.heap[j], s.depth)) {
                j++;
            }
            /* Exit if v is smaller than both sons */
            if (smaller(tree, v, s.heap[j], s.depth)) {
                break;
            }
            /* Exchange v with the smallest son */
            s.heap[k] = s.heap[j];
            k = j;
            /* And continue down the tree, setting j to the left son of k */
            j <<= 1;
        }
        s.heap[k] = v;
    }
    // inlined manually
    // var SMALLEST = 1;
    /* ===========================================================================
     * Send the block data compressed using the given Huffman trees
     */
    function compress_block(s, ltree, dtree) {
        var dist; /* distance of matched string */
        var lc; /* match length or unmatched char (if dist == 0) */
        var lx = 0; /* running index in l_buf */
        var code; /* the code to send */
        var extra; /* number of extra bits to send */
        if (s.last_lit !== 0) {
            do {
                dist = (s.pending_buf[s.d_buf + lx * 2] << 8) | (s.pending_buf[s.d_buf + lx * 2 + 1]);
                lc = s.pending_buf[s.l_buf + lx];
                lx++;
                if (dist === 0) {
                    send_code(s, lc, ltree); /* send a literal byte */
                    //Tracecv(isgraph(lc), (stderr," '%c' ", lc));
                }
                else {
                    /* Here, lc is the match length - MIN_MATCH */
                    code = _length_code[lc];
                    send_code(s, code + LITERALS + 1, ltree); /* send the length code */
                    extra = extra_lbits[code];
                    if (extra !== 0) {
                        lc -= base_length[code];
                        send_bits(s, lc, extra); /* send the extra length bits */
                    }
                    dist--; /* dist is now the match distance - 1 */
                    code = d_code(dist);
                    //Assert (code < D_CODES, "bad d_code");
                    send_code(s, code, dtree); /* send the distance code */
                    extra = extra_dbits[code];
                    if (extra !== 0) {
                        dist -= base_dist[code];
                        send_bits(s, dist, extra); /* send the extra distance bits */
                    }
                } /* literal or match pair ? */
                /* Check that the overlay between pending_buf and d_buf+l_buf is ok: */
                //Assert((uInt)(s->pending) < s->lit_bufsize + 2*lx,
                //       "pendingBuf overflow");
            } while (lx < s.last_lit);
        }
        send_code(s, END_BLOCK, ltree);
    }
    /* ===========================================================================
     * Construct one Huffman tree and assigns the code bit strings and lengths.
     * Update the total bit length for the current block.
     * IN assertion: the field freq is set for all tree elements.
     * OUT assertions: the fields len and code are set to the optimal bit length
     *     and corresponding code. The length opt_len is updated; static_len is
     *     also updated if stree is not null. The field max_code is set.
     */
    function build_tree(s, desc) {
        var tree = desc.dyn_tree;
        var stree = desc.stat_desc.static_tree;
        var has_stree = desc.stat_desc.has_stree;
        var elems = desc.stat_desc.elems;
        var n, m; /* iterate over heap elements */
        var max_code = -1; /* largest code with non zero frequency */
        var node; /* new node being created */
        /* Construct the initial heap, with least frequent element in
         * heap[SMALLEST]. The sons of heap[n] are heap[2*n] and heap[2*n+1].
         * heap[0] is not used.
         */
        s.heap_len = 0;
        s.heap_max = HEAP_SIZE;
        for (n = 0; n < elems; n++) {
            if (tree[n * 2] /*.Freq*/ !== 0) {
                s.heap[++s.heap_len] = max_code = n;
                s.depth[n] = 0;
            }
            else {
                tree[n * 2 + 1] /*.Len*/ = 0;
            }
        }
        /* The pkzip format requires that at least one distance code exists,
         * and that at least one bit should be sent even if there is only one
         * possible code. So to avoid special checks later on we force at least
         * two codes of non zero frequency.
         */
        while (s.heap_len < 2) {
            node = s.heap[++s.heap_len] = (max_code < 2 ? ++max_code : 0);
            tree[node * 2] /*.Freq*/ = 1;
            s.depth[node] = 0;
            s.opt_len--;
            if (has_stree) {
                s.static_len -= stree[node * 2 + 1] /*.Len*/;
            }
            /* node is 0 or 1 so it does not have extra bits */
        }
        desc.max_code = max_code;
        /* The elements heap[heap_len/2+1 .. heap_len] are leaves of the tree,
         * establish sub-heaps of increasing lengths:
         */
        for (n = (s.heap_len >> 1 /*int /2*/); n >= 1; n--) {
            pqdownheap(s, tree, n);
        }
        /* Construct the Huffman tree by repeatedly combining the least two
         * frequent nodes.
         */
        node = elems; /* next internal node of the tree */
        do {
            //pqremove(s, tree, n);  /* n = node of least frequency */
            /*** pqremove ***/
            n = s.heap[1 /*SMALLEST*/];
            s.heap[1 /*SMALLEST*/] = s.heap[s.heap_len--];
            pqdownheap(s, tree, 1 /*SMALLEST*/);
            /***/
            m = s.heap[1 /*SMALLEST*/]; /* m = node of next least frequency */
            s.heap[--s.heap_max] = n; /* keep the nodes sorted by frequency */
            s.heap[--s.heap_max] = m;
            /* Create a new node father of n and m */
            tree[node * 2] /*.Freq*/ = tree[n * 2] /*.Freq*/ + tree[m * 2] /*.Freq*/;
            s.depth[node] = (s.depth[n] >= s.depth[m] ? s.depth[n] : s.depth[m]) + 1;
            tree[n * 2 + 1] /*.Dad*/ = tree[m * 2 + 1] /*.Dad*/ = node;
            /* and insert the new node in the heap */
            s.heap[1 /*SMALLEST*/] = node++;
            pqdownheap(s, tree, 1 /*SMALLEST*/);
        } while (s.heap_len >= 2);
        s.heap[--s.heap_max] = s.heap[1 /*SMALLEST*/];
        /* At this point, the fields freq and dad are set. We can now
         * generate the bit lengths.
         */
        gen_bitlen(s, desc);
        /* The field len is now set, we can generate the bit codes */
        gen_codes(tree, max_code, s.bl_count);
    }
    /* ===========================================================================
     * Scan a literal or distance tree to determine the frequencies of the codes
     * in the bit length tree.
     */
    function scan_tree(s, tree, max_code) {
        var n; /* iterates over all tree elements */
        var prevlen = -1; /* last emitted length */
        var curlen; /* length of current code */
        var nextlen = tree[0 * 2 + 1] /*.Len*/; /* length of next code */
        var count = 0; /* repeat count of the current code */
        var max_count = 7; /* max repeat count */
        var min_count = 4; /* min repeat count */
        if (nextlen === 0) {
            max_count = 138;
            min_count = 3;
        }
        tree[(max_code + 1) * 2 + 1] /*.Len*/ = 0xffff; /* guard */
        for (n = 0; n <= max_code; n++) {
            curlen = nextlen;
            nextlen = tree[(n + 1) * 2 + 1] /*.Len*/;
            if (++count < max_count && curlen === nextlen) {
                continue;
            }
            else if (count < min_count) {
                s.bl_tree[curlen * 2] /*.Freq*/ += count;
            }
            else if (curlen !== 0) {
                if (curlen !== prevlen) {
                    s.bl_tree[curlen * 2] /*.Freq*/++;
                }
                s.bl_tree[REP_3_6 * 2] /*.Freq*/++;
            }
            else if (count <= 10) {
                s.bl_tree[REPZ_3_10 * 2] /*.Freq*/++;
            }
            else {
                s.bl_tree[REPZ_11_138 * 2] /*.Freq*/++;
            }
            count = 0;
            prevlen = curlen;
            if (nextlen === 0) {
                max_count = 138;
                min_count = 3;
            }
            else if (curlen === nextlen) {
                max_count = 6;
                min_count = 3;
            }
            else {
                max_count = 7;
                min_count = 4;
            }
        }
    }
    /* ===========================================================================
     * Send a literal or distance tree in compressed form, using the codes in
     * bl_tree.
     */
    function send_tree(s, tree, max_code) {
        var n; /* iterates over all tree elements */
        var prevlen = -1; /* last emitted length */
        var curlen; /* length of current code */
        var nextlen = tree[0 * 2 + 1] /*.Len*/; /* length of next code */
        var count = 0; /* repeat count of the current code */
        var max_count = 7; /* max repeat count */
        var min_count = 4; /* min repeat count */
        /* tree[max_code+1].Len = -1; */ /* guard already set */
        if (nextlen === 0) {
            max_count = 138;
            min_count = 3;
        }
        for (n = 0; n <= max_code; n++) {
            curlen = nextlen;
            nextlen = tree[(n + 1) * 2 + 1] /*.Len*/;
            if (++count < max_count && curlen === nextlen) {
                continue;
            }
            else if (count < min_count) {
                do {
                    send_code(s, curlen, s.bl_tree);
                } while (--count !== 0);
            }
            else if (curlen !== 0) {
                if (curlen !== prevlen) {
                    send_code(s, curlen, s.bl_tree);
                    count--;
                }
                //Assert(count >= 3 && count <= 6, " 3_6?");
                send_code(s, REP_3_6, s.bl_tree);
                send_bits(s, count - 3, 2);
            }
            else if (count <= 10) {
                send_code(s, REPZ_3_10, s.bl_tree);
                send_bits(s, count - 3, 3);
            }
            else {
                send_code(s, REPZ_11_138, s.bl_tree);
                send_bits(s, count - 11, 7);
            }
            count = 0;
            prevlen = curlen;
            if (nextlen === 0) {
                max_count = 138;
                min_count = 3;
            }
            else if (curlen === nextlen) {
                max_count = 6;
                min_count = 3;
            }
            else {
                max_count = 7;
                min_count = 4;
            }
        }
    }
    /* ===========================================================================
     * Construct the Huffman tree for the bit lengths and return the index in
     * bl_order of the last bit length code to send.
     */
    function build_bl_tree(s) {
        var max_blindex; /* index of last bit length code of non zero freq */
        /* Determine the bit length frequencies for literal and distance trees */
        scan_tree(s, s.dyn_ltree, s.l_desc.max_code);
        scan_tree(s, s.dyn_dtree, s.d_desc.max_code);
        /* Build the bit length tree: */
        build_tree(s, s.bl_desc);
        /* opt_len now includes the length of the tree representations, except
         * the lengths of the bit lengths codes and the 5+5+4 bits for the counts.
         */
        /* Determine the number of bit length codes to send. The pkzip format
         * requires that at least 4 bit length codes be sent. (appnote.txt says
         * 3 but the actual value used is 4.)
         */
        for (max_blindex = BL_CODES - 1; max_blindex >= 3; max_blindex--) {
            if (s.bl_tree[bl_order[max_blindex] * 2 + 1] /*.Len*/ !== 0) {
                break;
            }
        }
        /* Update opt_len to include the bit length tree and counts */
        s.opt_len += 3 * (max_blindex + 1) + 5 + 5 + 4;
        //Tracev((stderr, "\ndyn trees: dyn %ld, stat %ld",
        //        s->opt_len, s->static_len));
        return max_blindex;
    }
    /* ===========================================================================
     * Send the header for a block using dynamic Huffman trees: the counts, the
     * lengths of the bit length codes, the literal tree and the distance tree.
     * IN assertion: lcodes >= 257, dcodes >= 1, blcodes >= 4.
     */
    function send_all_trees(s, lcodes, dcodes, blcodes) {
        var rank; /* index in bl_order */
        //Assert (lcodes >= 257 && dcodes >= 1 && blcodes >= 4, "not enough codes");
        //Assert (lcodes <= L_CODES && dcodes <= D_CODES && blcodes <= BL_CODES,
        //        "too many codes");
        //Tracev((stderr, "\nbl counts: "));
        send_bits(s, lcodes - 257, 5); /* not +255 as stated in appnote.txt */
        send_bits(s, dcodes - 1, 5);
        send_bits(s, blcodes - 4, 4); /* not -3 as stated in appnote.txt */
        for (rank = 0; rank < blcodes; rank++) {
            //Tracev((stderr, "\nbl code %2d ", bl_order[rank]));
            send_bits(s, s.bl_tree[bl_order[rank] * 2 + 1] /*.Len*/, 3);
        }
        //Tracev((stderr, "\nbl tree: sent %ld", s->bits_sent));
        send_tree(s, s.dyn_ltree, lcodes - 1); /* literal tree */
        //Tracev((stderr, "\nlit tree: sent %ld", s->bits_sent));
        send_tree(s, s.dyn_dtree, dcodes - 1); /* distance tree */
        //Tracev((stderr, "\ndist tree: sent %ld", s->bits_sent));
    }
    /* ===========================================================================
     * Check if the data type is TEXT or BINARY, using the following algorithm:
     * - TEXT if the two conditions below are satisfied:
     *    a) There are no non-portable control characters belonging to the
     *       "black list" (0..6, 14..25, 28..31).
     *    b) There is at least one printable character belonging to the
     *       "white list" (9 {TAB}, 10 {LF}, 13 {CR}, 32..255).
     * - BINARY otherwise.
     * - The following partially-portable control characters form a
     *   "gray list" that is ignored in this detection algorithm:
     *   (7 {BEL}, 8 {BS}, 11 {VT}, 12 {FF}, 26 {SUB}, 27 {ESC}).
     * IN assertion: the fields Freq of dyn_ltree are set.
     */
    function detect_data_type(s) {
        /* black_mask is the bit mask of black-listed bytes
         * set bits 0..6, 14..25, and 28..31
         * 0xf3ffc07f = binary 11110011111111111100000001111111
         */
        var black_mask = 0xf3ffc07f;
        var n;
        /* Check for non-textual ("black-listed") bytes. */
        for (n = 0; n <= 31; n++, black_mask >>>= 1) {
            if ((black_mask & 1) && (s.dyn_ltree[n * 2] /*.Freq*/ !== 0)) {
                return Z_BINARY;
            }
        }
        /* Check for textual ("white-listed") bytes. */
        if (s.dyn_ltree[9 * 2] /*.Freq*/ !== 0 || s.dyn_ltree[10 * 2] /*.Freq*/ !== 0 ||
            s.dyn_ltree[13 * 2] /*.Freq*/ !== 0) {
            return Z_TEXT;
        }
        for (n = 32; n < LITERALS; n++) {
            if (s.dyn_ltree[n * 2] /*.Freq*/ !== 0) {
                return Z_TEXT;
            }
        }
        /* There are no "black-listed" or "white-listed" bytes:
         * this stream either is empty or has tolerated ("gray-listed") bytes only.
         */
        return Z_BINARY;
    }
    var static_init_done = false;
    /* ===========================================================================
     * Initialize the tree data structures for a new zlib stream.
     */
    function _tr_init(s) {
        if (!static_init_done) {
            tr_static_init();
            static_init_done = true;
        }
        s.l_desc = new TreeDesc(s.dyn_ltree, static_l_desc);
        s.d_desc = new TreeDesc(s.dyn_dtree, static_d_desc);
        s.bl_desc = new TreeDesc(s.bl_tree, static_bl_desc);
        s.bi_buf = 0;
        s.bi_valid = 0;
        /* Initialize the first block of the first file: */
        init_block(s);
    }
    /* ===========================================================================
     * Send a stored block
     */
    function _tr_stored_block(s, buf, stored_len, last) {
        send_bits(s, (STORED_BLOCK << 1) + (last ? 1 : 0), 3); /* send block type */
        copy_block(s, buf, stored_len, true); /* with header */
    }
    /* ===========================================================================
     * Send one empty static block to give enough lookahead for inflate.
     * This takes 10 bits, of which 7 may remain in the bit buffer.
     */
    function _tr_align(s) {
        send_bits(s, STATIC_TREES << 1, 3);
        send_code(s, END_BLOCK, static_ltree);
        bi_flush(s);
    }
    /* ===========================================================================
     * Determine the best encoding for the current block: dynamic trees, static
     * trees or store, and output the encoded block to the zip file.
     */
    function _tr_flush_block(s, buf, stored_len, last) {
        var opt_lenb, static_lenb; /* opt_len and static_len in bytes */
        var max_blindex = 0; /* index of last bit length code of non zero freq */
        /* Build the Huffman trees unless a stored block is forced */
        if (s.level > 0) {
            /* Check if the file is binary or text */
            if (s.strm.data_type === Z_UNKNOWN) {
                s.strm.data_type = detect_data_type(s);
            }
            /* Construct the literal and distance trees */
            build_tree(s, s.l_desc);
            // Tracev((stderr, "\nlit data: dyn %ld, stat %ld", s->opt_len,
            //        s->static_len));
            build_tree(s, s.d_desc);
            // Tracev((stderr, "\ndist data: dyn %ld, stat %ld", s->opt_len,
            //        s->static_len));
            /* At this point, opt_len and static_len are the total bit lengths of
             * the compressed block data, excluding the tree representations.
             */
            /* Build the bit length tree for the above two trees, and get the index
             * in bl_order of the last bit length code to send.
             */
            max_blindex = build_bl_tree(s);
            /* Determine the best encoding. Compute the block lengths in bytes. */
            opt_lenb = (s.opt_len + 3 + 7) >>> 3;
            static_lenb = (s.static_len + 3 + 7) >>> 3;
            // Tracev((stderr, "\nopt %lu(%lu) stat %lu(%lu) stored %lu lit %u ",
            //        opt_lenb, s->opt_len, static_lenb, s->static_len, stored_len,
            //        s->last_lit));
            if (static_lenb <= opt_lenb) {
                opt_lenb = static_lenb;
            }
        }
        else {
            // Assert(buf != (char*)0, "lost buf");
            opt_lenb = static_lenb = stored_len + 5; /* force a stored block */
        }
        if ((stored_len + 4 <= opt_lenb) && (buf !== -1)) {
            /* 4: two words for the lengths */
            /* The test buf != NULL is only necessary if LIT_BUFSIZE > WSIZE.
             * Otherwise we can't have processed more than WSIZE input bytes since
             * the last block flush, because compression would have been
             * successful. If LIT_BUFSIZE <= WSIZE, it is never too late to
             * transform a block into a stored block.
             */
            _tr_stored_block(s, buf, stored_len, last);
        }
        else if (s.strategy === Z_FIXED || static_lenb === opt_lenb) {
            send_bits(s, (STATIC_TREES << 1) + (last ? 1 : 0), 3);
            compress_block(s, static_ltree, static_dtree);
        }
        else {
            send_bits(s, (DYN_TREES << 1) + (last ? 1 : 0), 3);
            send_all_trees(s, s.l_desc.max_code + 1, s.d_desc.max_code + 1, max_blindex + 1);
            compress_block(s, s.dyn_ltree, s.dyn_dtree);
        }
        // Assert (s->compressed_len == s->bits_sent, "bad compressed size");
        /* The above check is made mod 2^32, for files larger than 512 MB
         * and uLong implemented on 32 bits.
         */
        init_block(s);
        if (last) {
            bi_windup(s);
        }
        // Tracev((stderr,"\ncomprlen %lu(%lu) ", s->compressed_len>>3,
        //       s->compressed_len-7*last));
    }
    /* ===========================================================================
     * Save the match info and tally the frequency counts. Return true if
     * the current block must be flushed.
     */
    function _tr_tally(s, dist, lc) {
        //var out_length, in_length, dcode;
        s.pending_buf[s.d_buf + s.last_lit * 2] = (dist >>> 8) & 0xff;
        s.pending_buf[s.d_buf + s.last_lit * 2 + 1] = dist & 0xff;
        s.pending_buf[s.l_buf + s.last_lit] = lc & 0xff;
        s.last_lit++;
        if (dist === 0) {
            /* lc is the unmatched char */
            s.dyn_ltree[lc * 2] /*.Freq*/++;
        }
        else {
            s.matches++;
            /* Here, lc is the match length - MIN_MATCH */
            dist--; /* dist = match distance - 1 */
            //Assert((ush)dist < (ush)MAX_DIST(s) &&
            //       (ush)lc <= (ush)(MAX_MATCH-MIN_MATCH) &&
            //       (ush)d_code(dist) < (ush)D_CODES,  "_tr_tally: bad match");
            s.dyn_ltree[(_length_code[lc] + LITERALS + 1) * 2] /*.Freq*/++;
            s.dyn_dtree[d_code(dist) * 2] /*.Freq*/++;
        }
        // (!) This block is disabled in zlib defaults,
        // don't enable it for binary compatibility
        //#ifdef TRUNCATE_BLOCK
        //  /* Try to guess if it is profitable to stop the current block here */
        //  if ((s.last_lit & 0x1fff) === 0 && s.level > 2) {
        //    /* Compute an upper bound for the compressed length */
        //    out_length = s.last_lit*8;
        //    in_length = s.strstart - s.block_start;
        //
        //    for (dcode = 0; dcode < D_CODES; dcode++) {
        //      out_length += s.dyn_dtree[dcode*2]/*.Freq*/ * (5 + extra_dbits[dcode]);
        //    }
        //    out_length >>>= 3;
        //    //Tracev((stderr,"\nlast_lit %u, in %ld, out ~%ld(%ld%%) ",
        //    //       s->last_lit, in_length, out_length,
        //    //       100L - out_length*100L/in_length));
        //    if (s.matches < (s.last_lit>>1)/*int /2*/ && out_length < (in_length>>1)/*int /2*/) {
        //      return true;
        //    }
        //  }
        //#endif
        return (s.last_lit === s.lit_bufsize - 1);
        /* We avoid equality with lit_bufsize because of wraparound at 64K
         * on 16 bit machines and because stored blocks are restricted to
         * 64K-1 bytes.
         */
    }
    var _tr_init_1 = _tr_init;
    var _tr_stored_block_1 = _tr_stored_block;
    var _tr_flush_block_1 = _tr_flush_block;
    var _tr_tally_1 = _tr_tally;
    var _tr_align_1 = _tr_align;
    var trees = {
        _tr_init: _tr_init_1,
        _tr_stored_block: _tr_stored_block_1,
        _tr_flush_block: _tr_flush_block_1,
        _tr_tally: _tr_tally_1,
        _tr_align: _tr_align_1
    };

    var trees$1 = /*#__PURE__*/Object.freeze({
        default: trees,
        __moduleExports: trees,
        _tr_init: _tr_init_1,
        _tr_stored_block: _tr_stored_block_1,
        _tr_flush_block: _tr_flush_block_1,
        _tr_tally: _tr_tally_1,
        _tr_align: _tr_align_1
    });

    // Note: adler32 takes 12% for level 0 and 2% for level 6.
    // It isn't worth it to make additional optimizations as in original.
    // Small size is preferable.
    // (C) 1995-2013 Jean-loup Gailly and Mark Adler
    // (C) 2014-2017 Vitaly Puzrin and Andrey Tupitsin
    //
    // This software is provided 'as-is', without any express or implied
    // warranty. In no event will the authors be held liable for any damages
    // arising from the use of this software.
    //
    // Permission is granted to anyone to use this software for any purpose,
    // including commercial applications, and to alter it and redistribute it
    // freely, subject to the following restrictions:
    //
    // 1. The origin of this software must not be misrepresented; you must not
    //   claim that you wrote the original software. If you use this software
    //   in a product, an acknowledgment in the product documentation would be
    //   appreciated but is not required.
    // 2. Altered source versions must be plainly marked as such, and must not be
    //   misrepresented as being the original software.
    // 3. This notice may not be removed or altered from any source distribution.
    function adler32(adler, buf, len, pos) {
        var s1 = (adler & 0xffff) | 0, s2 = ((adler >>> 16) & 0xffff) | 0, n = 0;
        while (len !== 0) {
            // Set limit ~ twice less than 5552, to keep
            // s2 in 31-bits, because we force signed ints.
            // in other case %= will fail.
            n = len > 2000 ? 2000 : len;
            len -= n;
            do {
                s1 = (s1 + buf[pos++]) | 0;
                s2 = (s2 + s1) | 0;
            } while (--n);
            s1 %= 65521;
            s2 %= 65521;
        }
        return (s1 | (s2 << 16)) | 0;
    }
    var adler32_1 = adler32;

    var adler32$1 = /*#__PURE__*/Object.freeze({
        default: adler32_1,
        __moduleExports: adler32_1
    });

    // Note: we can't get significant speed boost here.
    // So write code to minimize size - no pregenerated tables
    // and array tools dependencies.
    // (C) 1995-2013 Jean-loup Gailly and Mark Adler
    // (C) 2014-2017 Vitaly Puzrin and Andrey Tupitsin
    //
    // This software is provided 'as-is', without any express or implied
    // warranty. In no event will the authors be held liable for any damages
    // arising from the use of this software.
    //
    // Permission is granted to anyone to use this software for any purpose,
    // including commercial applications, and to alter it and redistribute it
    // freely, subject to the following restrictions:
    //
    // 1. The origin of this software must not be misrepresented; you must not
    //   claim that you wrote the original software. If you use this software
    //   in a product, an acknowledgment in the product documentation would be
    //   appreciated but is not required.
    // 2. Altered source versions must be plainly marked as such, and must not be
    //   misrepresented as being the original software.
    // 3. This notice may not be removed or altered from any source distribution.
    // Use ordinary array, since untyped makes no boost here
    function makeTable() {
        var c, table = [];
        for (var n = 0; n < 256; n++) {
            c = n;
            for (var k = 0; k < 8; k++) {
                c = ((c & 1) ? (0xEDB88320 ^ (c >>> 1)) : (c >>> 1));
            }
            table[n] = c;
        }
        return table;
    }
    // Create table on load. Just 255 signed longs. Not a problem.
    var crcTable = makeTable();
    function crc32(crc, buf, len, pos) {
        var t = crcTable, end = pos + len;
        crc ^= -1;
        for (var i = pos; i < end; i++) {
            crc = (crc >>> 8) ^ t[(crc ^ buf[i]) & 0xFF];
        }
        return (crc ^ (-1)); // >>> 0;
    }
    var crc32_1 = crc32;

    var crc32$1 = /*#__PURE__*/Object.freeze({
        default: crc32_1,
        __moduleExports: crc32_1
    });

    // (C) 1995-2013 Jean-loup Gailly and Mark Adler
    // (C) 2014-2017 Vitaly Puzrin and Andrey Tupitsin
    //
    // This software is provided 'as-is', without any express or implied
    // warranty. In no event will the authors be held liable for any damages
    // arising from the use of this software.
    //
    // Permission is granted to anyone to use this software for any purpose,
    // including commercial applications, and to alter it and redistribute it
    // freely, subject to the following restrictions:
    //
    // 1. The origin of this software must not be misrepresented; you must not
    //   claim that you wrote the original software. If you use this software
    //   in a product, an acknowledgment in the product documentation would be
    //   appreciated but is not required.
    // 2. Altered source versions must be plainly marked as such, and must not be
    //   misrepresented as being the original software.
    // 3. This notice may not be removed or altered from any source distribution.
    var messages = {
        2: 'need dictionary',
        1: 'stream end',
        0: '',
        '-1': 'file error',
        '-2': 'stream error',
        '-3': 'data error',
        '-4': 'insufficient memory',
        '-5': 'buffer error',
        '-6': 'incompatible version' /* Z_VERSION_ERROR (-6) */
    };

    var messages$1 = /*#__PURE__*/Object.freeze({
        default: messages,
        __moduleExports: messages
    });

    var trees$2 = (trees$1 && trees) || trees$1;

    var adler32$2 = (adler32$1 && adler32_1) || adler32$1;

    var crc32$2 = (crc32$1 && crc32_1) || crc32$1;

    var msg = (messages$1 && messages) || messages$1;

    // (C) 1995-2013 Jean-loup Gailly and Mark Adler
    // (C) 2014-2017 Vitaly Puzrin and Andrey Tupitsin
    //
    // This software is provided 'as-is', without any express or implied
    // warranty. In no event will the authors be held liable for any damages
    // arising from the use of this software.
    //
    // Permission is granted to anyone to use this software for any purpose,
    // including commercial applications, and to alter it and redistribute it
    // freely, subject to the following restrictions:
    //
    // 1. The origin of this software must not be misrepresented; you must not
    //   claim that you wrote the original software. If you use this software
    //   in a product, an acknowledgment in the product documentation would be
    //   appreciated but is not required.
    // 2. Altered source versions must be plainly marked as such, and must not be
    //   misrepresented as being the original software.
    // 3. This notice may not be removed or altered from any source distribution.
    /* Public constants ==========================================================*/
    /* ===========================================================================*/
    /* Allowed flush values; see deflate() and inflate() below for details */
    var Z_NO_FLUSH = 0;
    var Z_PARTIAL_FLUSH = 1;
    //var Z_SYNC_FLUSH    = 2;
    var Z_FULL_FLUSH = 3;
    var Z_FINISH = 4;
    var Z_BLOCK = 5;
    //var Z_TREES         = 6;
    /* Return codes for the compression/decompression functions. Negative values
     * are errors, positive values are used for special but normal events.
     */
    var Z_OK = 0;
    var Z_STREAM_END = 1;
    //var Z_NEED_DICT     = 2;
    //var Z_ERRNO         = -1;
    var Z_STREAM_ERROR = -2;
    var Z_DATA_ERROR = -3;
    //var Z_MEM_ERROR     = -4;
    var Z_BUF_ERROR = -5;
    //var Z_VERSION_ERROR = -6;
    /* compression levels */
    //var Z_NO_COMPRESSION      = 0;
    //var Z_BEST_SPEED          = 1;
    //var Z_BEST_COMPRESSION    = 9;
    var Z_DEFAULT_COMPRESSION = -1;
    var Z_FILTERED = 1;
    var Z_HUFFMAN_ONLY = 2;
    var Z_RLE = 3;
    var Z_FIXED$1 = 4;
    var Z_DEFAULT_STRATEGY = 0;
    /* Possible values of the data_type field (though see inflate()) */
    //var Z_BINARY              = 0;
    //var Z_TEXT                = 1;
    //var Z_ASCII               = 1; // = Z_TEXT
    var Z_UNKNOWN$1 = 2;
    /* The deflate compression method */
    var Z_DEFLATED = 8;
    /*============================================================================*/
    var MAX_MEM_LEVEL = 9;
    /* Maximum value for memLevel in deflateInit2 */
    var MAX_WBITS = 15;
    /* 32K LZ77 window */
    var DEF_MEM_LEVEL = 8;
    var LENGTH_CODES$1 = 29;
    /* number of length codes, not counting the special END_BLOCK code */
    var LITERALS$1 = 256;
    /* number of literal bytes 0..255 */
    var L_CODES$1 = LITERALS$1 + 1 + LENGTH_CODES$1;
    /* number of Literal or Length codes, including the END_BLOCK code */
    var D_CODES$1 = 30;
    /* number of distance codes */
    var BL_CODES$1 = 19;
    /* number of codes used to transfer the bit lengths */
    var HEAP_SIZE$1 = 2 * L_CODES$1 + 1;
    /* maximum heap size */
    var MAX_BITS$1 = 15;
    /* All codes must not exceed MAX_BITS bits */
    var MIN_MATCH$1 = 3;
    var MAX_MATCH$1 = 258;
    var MIN_LOOKAHEAD = (MAX_MATCH$1 + MIN_MATCH$1 + 1);
    var PRESET_DICT = 0x20;
    var INIT_STATE = 42;
    var EXTRA_STATE = 69;
    var NAME_STATE = 73;
    var COMMENT_STATE = 91;
    var HCRC_STATE = 103;
    var BUSY_STATE = 113;
    var FINISH_STATE = 666;
    var BS_NEED_MORE = 1; /* block not completed, need more input or more output */
    var BS_BLOCK_DONE = 2; /* block flush performed */
    var BS_FINISH_STARTED = 3; /* finish started, need only more output at next deflate */
    var BS_FINISH_DONE = 4; /* finish done, accept no more input or output */
    var OS_CODE = 0x03; // Unix :) . Don't detect, use this default.
    function err(strm, errorCode) {
        strm.msg = msg[errorCode];
        return errorCode;
    }
    function rank(f) {
        return ((f) << 1) - ((f) > 4 ? 9 : 0);
    }
    function zero$1(buf) { var len = buf.length; while (--len >= 0) {
        buf[len] = 0;
    } }
    /* =========================================================================
     * Flush as much pending output as possible. All deflate() output goes
     * through this function so some applications may wish to modify it
     * to avoid allocating a large strm->output buffer and copying into it.
     * (See also read_buf()).
     */
    function flush_pending(strm) {
        var s = strm.state;
        //_tr_flush_bits(s);
        var len = s.pending;
        if (len > strm.avail_out) {
            len = strm.avail_out;
        }
        if (len === 0) {
            return;
        }
        utils.arraySet(strm.output, s.pending_buf, s.pending_out, len, strm.next_out);
        strm.next_out += len;
        s.pending_out += len;
        strm.total_out += len;
        strm.avail_out -= len;
        s.pending -= len;
        if (s.pending === 0) {
            s.pending_out = 0;
        }
    }
    function flush_block_only(s, last) {
        trees$2._tr_flush_block(s, (s.block_start >= 0 ? s.block_start : -1), s.strstart - s.block_start, last);
        s.block_start = s.strstart;
        flush_pending(s.strm);
    }
    function put_byte(s, b) {
        s.pending_buf[s.pending++] = b;
    }
    /* =========================================================================
     * Put a short in the pending buffer. The 16-bit value is put in MSB order.
     * IN assertion: the stream state is correct and there is enough room in
     * pending_buf.
     */
    function putShortMSB(s, b) {
        //  put_byte(s, (Byte)(b >> 8));
        //  put_byte(s, (Byte)(b & 0xff));
        s.pending_buf[s.pending++] = (b >>> 8) & 0xff;
        s.pending_buf[s.pending++] = b & 0xff;
    }
    /* ===========================================================================
     * Read a new buffer from the current input stream, update the adler32
     * and total number of bytes read.  All deflate() input goes through
     * this function so some applications may wish to modify it to avoid
     * allocating a large strm->input buffer and copying from it.
     * (See also flush_pending()).
     */
    function read_buf(strm, buf, start, size) {
        var len = strm.avail_in;
        if (len > size) {
            len = size;
        }
        if (len === 0) {
            return 0;
        }
        strm.avail_in -= len;
        // zmemcpy(buf, strm->next_in, len);
        utils.arraySet(buf, strm.input, strm.next_in, len, start);
        if (strm.state.wrap === 1) {
            strm.adler = adler32$2(strm.adler, buf, len, start);
        }
        else if (strm.state.wrap === 2) {
            strm.adler = crc32$2(strm.adler, buf, len, start);
        }
        strm.next_in += len;
        strm.total_in += len;
        return len;
    }
    /* ===========================================================================
     * Set match_start to the longest match starting at the given string and
     * return its length. Matches shorter or equal to prev_length are discarded,
     * in which case the result is equal to prev_length and match_start is
     * garbage.
     * IN assertions: cur_match is the head of the hash chain for the current
     *   string (strstart) and its distance is <= MAX_DIST, and prev_length >= 1
     * OUT assertion: the match length is not greater than s->lookahead.
     */
    function longest_match(s, cur_match) {
        var chain_length = s.max_chain_length; /* max hash chain length */
        var scan = s.strstart; /* current string */
        var match; /* matched string */
        var len; /* length of current match */
        var best_len = s.prev_length; /* best match length so far */
        var nice_match = s.nice_match; /* stop if match long enough */
        var limit = (s.strstart > (s.w_size - MIN_LOOKAHEAD)) ?
            s.strstart - (s.w_size - MIN_LOOKAHEAD) : 0 /*NIL*/;
        var _win = s.window; // shortcut
        var wmask = s.w_mask;
        var prev = s.prev;
        /* Stop when cur_match becomes <= limit. To simplify the code,
         * we prevent matches with the string of window index 0.
         */
        var strend = s.strstart + MAX_MATCH$1;
        var scan_end1 = _win[scan + best_len - 1];
        var scan_end = _win[scan + best_len];
        /* The code is optimized for HASH_BITS >= 8 and MAX_MATCH-2 multiple of 16.
         * It is easy to get rid of this optimization if necessary.
         */
        // Assert(s->hash_bits >= 8 && MAX_MATCH == 258, "Code too clever");
        /* Do not waste too much time if we already have a good match: */
        if (s.prev_length >= s.good_match) {
            chain_length >>= 2;
        }
        /* Do not look for matches beyond the end of the input. This is necessary
         * to make deflate deterministic.
         */
        if (nice_match > s.lookahead) {
            nice_match = s.lookahead;
        }
        // Assert((ulg)s->strstart <= s->window_size-MIN_LOOKAHEAD, "need lookahead");
        do {
            // Assert(cur_match < s->strstart, "no future");
            match = cur_match;
            /* Skip to next match if the match length cannot increase
             * or if the match length is less than 2.  Note that the checks below
             * for insufficient lookahead only occur occasionally for performance
             * reasons.  Therefore uninitialized memory will be accessed, and
             * conditional jumps will be made that depend on those values.
             * However the length of the match is limited to the lookahead, so
             * the output of deflate is not affected by the uninitialized values.
             */
            if (_win[match + best_len] !== scan_end ||
                _win[match + best_len - 1] !== scan_end1 ||
                _win[match] !== _win[scan] ||
                _win[++match] !== _win[scan + 1]) {
                continue;
            }
            /* The check at best_len-1 can be removed because it will be made
             * again later. (This heuristic is not always a win.)
             * It is not necessary to compare scan[2] and match[2] since they
             * are always equal when the other bytes match, given that
             * the hash keys are equal and that HASH_BITS >= 8.
             */
            scan += 2;
            match++;
            // Assert(*scan == *match, "match[2]?");
            /* We check for insufficient lookahead only every 8th comparison;
             * the 256th check will be made at strstart+258.
             */
            do {
                /*jshint noempty:false*/
            } while (_win[++scan] === _win[++match] && _win[++scan] === _win[++match] &&
                _win[++scan] === _win[++match] && _win[++scan] === _win[++match] &&
                _win[++scan] === _win[++match] && _win[++scan] === _win[++match] &&
                _win[++scan] === _win[++match] && _win[++scan] === _win[++match] &&
                scan < strend);
            // Assert(scan <= s->window+(unsigned)(s->window_size-1), "wild scan");
            len = MAX_MATCH$1 - (strend - scan);
            scan = strend - MAX_MATCH$1;
            if (len > best_len) {
                s.match_start = cur_match;
                best_len = len;
                if (len >= nice_match) {
                    break;
                }
                scan_end1 = _win[scan + best_len - 1];
                scan_end = _win[scan + best_len];
            }
        } while ((cur_match = prev[cur_match & wmask]) > limit && --chain_length !== 0);
        if (best_len <= s.lookahead) {
            return best_len;
        }
        return s.lookahead;
    }
    /* ===========================================================================
     * Fill the window when the lookahead becomes insufficient.
     * Updates strstart and lookahead.
     *
     * IN assertion: lookahead < MIN_LOOKAHEAD
     * OUT assertions: strstart <= window_size-MIN_LOOKAHEAD
     *    At least one byte has been read, or avail_in == 0; reads are
     *    performed for at least two bytes (required for the zip translate_eol
     *    option -- not supported here).
     */
    function fill_window(s) {
        var _w_size = s.w_size;
        var p, n, m, more, str;
        //Assert(s->lookahead < MIN_LOOKAHEAD, "already enough lookahead");
        do {
            more = s.window_size - s.lookahead - s.strstart;
            // JS ints have 32 bit, block below not needed
            /* Deal with !@#$% 64K limit: */
            //if (sizeof(int) <= 2) {
            //    if (more == 0 && s->strstart == 0 && s->lookahead == 0) {
            //        more = wsize;
            //
            //  } else if (more == (unsigned)(-1)) {
            //        /* Very unlikely, but possible on 16 bit machine if
            //         * strstart == 0 && lookahead == 1 (input done a byte at time)
            //         */
            //        more--;
            //    }
            //}
            /* If the window is almost full and there is insufficient lookahead,
             * move the upper half to the lower one to make room in the upper half.
             */
            if (s.strstart >= _w_size + (_w_size - MIN_LOOKAHEAD)) {
                utils.arraySet(s.window, s.window, _w_size, _w_size, 0);
                s.match_start -= _w_size;
                s.strstart -= _w_size;
                /* we now have strstart >= MAX_DIST */
                s.block_start -= _w_size;
                /* Slide the hash table (could be avoided with 32 bit values
                 at the expense of memory usage). We slide even when level == 0
                 to keep the hash table consistent if we switch back to level > 0
                 later. (Using level 0 permanently is not an optimal usage of
                 zlib, so we don't care about this pathological case.)
                 */
                n = s.hash_size;
                p = n;
                do {
                    m = s.head[--p];
                    s.head[p] = (m >= _w_size ? m - _w_size : 0);
                } while (--n);
                n = _w_size;
                p = n;
                do {
                    m = s.prev[--p];
                    s.prev[p] = (m >= _w_size ? m - _w_size : 0);
                    /* If n is not on any hash chain, prev[n] is garbage but
                     * its value will never be used.
                     */
                } while (--n);
                more += _w_size;
            }
            if (s.strm.avail_in === 0) {
                break;
            }
            /* If there was no sliding:
             *    strstart <= WSIZE+MAX_DIST-1 && lookahead <= MIN_LOOKAHEAD - 1 &&
             *    more == window_size - lookahead - strstart
             * => more >= window_size - (MIN_LOOKAHEAD-1 + WSIZE + MAX_DIST-1)
             * => more >= window_size - 2*WSIZE + 2
             * In the BIG_MEM or MMAP case (not yet supported),
             *   window_size == input_size + MIN_LOOKAHEAD  &&
             *   strstart + s->lookahead <= input_size => more >= MIN_LOOKAHEAD.
             * Otherwise, window_size == 2*WSIZE so more >= 2.
             * If there was sliding, more >= WSIZE. So in all cases, more >= 2.
             */
            //Assert(more >= 2, "more < 2");
            n = read_buf(s.strm, s.window, s.strstart + s.lookahead, more);
            s.lookahead += n;
            /* Initialize the hash value now that we have some input: */
            if (s.lookahead + s.insert >= MIN_MATCH$1) {
                str = s.strstart - s.insert;
                s.ins_h = s.window[str];
                /* UPDATE_HASH(s, s->ins_h, s->window[str + 1]); */
                s.ins_h = ((s.ins_h << s.hash_shift) ^ s.window[str + 1]) & s.hash_mask;
                //#if MIN_MATCH != 3
                //        Call update_hash() MIN_MATCH-3 more times
                //#endif
                while (s.insert) {
                    /* UPDATE_HASH(s, s->ins_h, s->window[str + MIN_MATCH-1]); */
                    s.ins_h = ((s.ins_h << s.hash_shift) ^ s.window[str + MIN_MATCH$1 - 1]) & s.hash_mask;
                    s.prev[str & s.w_mask] = s.head[s.ins_h];
                    s.head[s.ins_h] = str;
                    str++;
                    s.insert--;
                    if (s.lookahead + s.insert < MIN_MATCH$1) {
                        break;
                    }
                }
            }
            /* If the whole input has less than MIN_MATCH bytes, ins_h is garbage,
             * but this is not important since only literal bytes will be emitted.
             */
        } while (s.lookahead < MIN_LOOKAHEAD && s.strm.avail_in !== 0);
        /* If the WIN_INIT bytes after the end of the current data have never been
         * written, then zero those bytes in order to avoid memory check reports of
         * the use of uninitialized (or uninitialised as Julian writes) bytes by
         * the longest match routines.  Update the high water mark for the next
         * time through here.  WIN_INIT is set to MAX_MATCH since the longest match
         * routines allow scanning to strstart + MAX_MATCH, ignoring lookahead.
         */
        //  if (s.high_water < s.window_size) {
        //    var curr = s.strstart + s.lookahead;
        //    var init = 0;
        //
        //    if (s.high_water < curr) {
        //      /* Previous high water mark below current data -- zero WIN_INIT
        //       * bytes or up to end of window, whichever is less.
        //       */
        //      init = s.window_size - curr;
        //      if (init > WIN_INIT)
        //        init = WIN_INIT;
        //      zmemzero(s->window + curr, (unsigned)init);
        //      s->high_water = curr + init;
        //    }
        //    else if (s->high_water < (ulg)curr + WIN_INIT) {
        //      /* High water mark at or above current data, but below current data
        //       * plus WIN_INIT -- zero out to current data plus WIN_INIT, or up
        //       * to end of window, whichever is less.
        //       */
        //      init = (ulg)curr + WIN_INIT - s->high_water;
        //      if (init > s->window_size - s->high_water)
        //        init = s->window_size - s->high_water;
        //      zmemzero(s->window + s->high_water, (unsigned)init);
        //      s->high_water += init;
        //    }
        //  }
        //
        //  Assert((ulg)s->strstart <= s->window_size - MIN_LOOKAHEAD,
        //    "not enough room for search");
    }
    /* ===========================================================================
     * Copy without compression as much as possible from the input stream, return
     * the current block state.
     * This function does not insert new strings in the dictionary since
     * uncompressible data is probably not useful. This function is used
     * only for the level=0 compression option.
     * NOTE: this function should be optimized to avoid extra copying from
     * window to pending_buf.
     */
    function deflate_stored(s, flush) {
        /* Stored blocks are limited to 0xffff bytes, pending_buf is limited
         * to pending_buf_size, and each stored block has a 5 byte header:
         */
        var max_block_size = 0xffff;
        if (max_block_size > s.pending_buf_size - 5) {
            max_block_size = s.pending_buf_size - 5;
        }
        /* Copy as much as possible from input to output: */
        for (;;) {
            /* Fill the window as much as possible: */
            if (s.lookahead <= 1) {
                //Assert(s->strstart < s->w_size+MAX_DIST(s) ||
                //  s->block_start >= (long)s->w_size, "slide too late");
                //      if (!(s.strstart < s.w_size + (s.w_size - MIN_LOOKAHEAD) ||
                //        s.block_start >= s.w_size)) {
                //        throw  new Error("slide too late");
                //      }
                fill_window(s);
                if (s.lookahead === 0 && flush === Z_NO_FLUSH) {
                    return BS_NEED_MORE;
                }
                if (s.lookahead === 0) {
                    break;
                }
                /* flush the current block */
            }
            //Assert(s->block_start >= 0L, "block gone");
            //    if (s.block_start < 0) throw new Error("block gone");
            s.strstart += s.lookahead;
            s.lookahead = 0;
            /* Emit a stored block if pending_buf will be full: */
            var max_start = s.block_start + max_block_size;
            if (s.strstart === 0 || s.strstart >= max_start) {
                /* strstart == 0 is possible when wraparound on 16-bit machine */
                s.lookahead = s.strstart - max_start;
                s.strstart = max_start;
                /*** FLUSH_BLOCK(s, 0); ***/
                flush_block_only(s, false);
                if (s.strm.avail_out === 0) {
                    return BS_NEED_MORE;
                }
                /***/
            }
            /* Flush if we may have to slide, otherwise block_start may become
             * negative and the data will be gone:
             */
            if (s.strstart - s.block_start >= (s.w_size - MIN_LOOKAHEAD)) {
                /*** FLUSH_BLOCK(s, 0); ***/
                flush_block_only(s, false);
                if (s.strm.avail_out === 0) {
                    return BS_NEED_MORE;
                }
                /***/
            }
        }
        s.insert = 0;
        if (flush === Z_FINISH) {
            /*** FLUSH_BLOCK(s, 1); ***/
            flush_block_only(s, true);
            if (s.strm.avail_out === 0) {
                return BS_FINISH_STARTED;
            }
            /***/
            return BS_FINISH_DONE;
        }
        if (s.strstart > s.block_start) {
            /*** FLUSH_BLOCK(s, 0); ***/
            flush_block_only(s, false);
            if (s.strm.avail_out === 0) {
                return BS_NEED_MORE;
            }
            /***/
        }
        return BS_NEED_MORE;
    }
    /* ===========================================================================
     * Compress as much as possible from the input stream, return the current
     * block state.
     * This function does not perform lazy evaluation of matches and inserts
     * new strings in the dictionary only for unmatched strings or for short
     * matches. It is used only for the fast compression options.
     */
    function deflate_fast(s, flush) {
        var hash_head; /* head of the hash chain */
        var bflush; /* set if current block must be flushed */
        for (;;) {
            /* Make sure that we always have enough lookahead, except
             * at the end of the input file. We need MAX_MATCH bytes
             * for the next match, plus MIN_MATCH bytes to insert the
             * string following the next match.
             */
            if (s.lookahead < MIN_LOOKAHEAD) {
                fill_window(s);
                if (s.lookahead < MIN_LOOKAHEAD && flush === Z_NO_FLUSH) {
                    return BS_NEED_MORE;
                }
                if (s.lookahead === 0) {
                    break; /* flush the current block */
                }
            }
            /* Insert the string window[strstart .. strstart+2] in the
             * dictionary, and set hash_head to the head of the hash chain:
             */
            hash_head = 0 /*NIL*/;
            if (s.lookahead >= MIN_MATCH$1) {
                /*** INSERT_STRING(s, s.strstart, hash_head); ***/
                s.ins_h = ((s.ins_h << s.hash_shift) ^ s.window[s.strstart + MIN_MATCH$1 - 1]) & s.hash_mask;
                hash_head = s.prev[s.strstart & s.w_mask] = s.head[s.ins_h];
                s.head[s.ins_h] = s.strstart;
                /***/
            }
            /* Find the longest match, discarding those <= prev_length.
             * At this point we have always match_length < MIN_MATCH
             */
            if (hash_head !== 0 /*NIL*/ && ((s.strstart - hash_head) <= (s.w_size - MIN_LOOKAHEAD))) {
                /* To simplify the code, we prevent matches with the string
                 * of window index 0 (in particular we have to avoid a match
                 * of the string with itself at the start of the input file).
                 */
                s.match_length = longest_match(s, hash_head);
                /* longest_match() sets match_start */
            }
            if (s.match_length >= MIN_MATCH$1) {
                // check_match(s, s.strstart, s.match_start, s.match_length); // for debug only
                /*** _tr_tally_dist(s, s.strstart - s.match_start,
                               s.match_length - MIN_MATCH, bflush); ***/
                bflush = trees$2._tr_tally(s, s.strstart - s.match_start, s.match_length - MIN_MATCH$1);
                s.lookahead -= s.match_length;
                /* Insert new strings in the hash table only if the match length
                 * is not too large. This saves time but degrades compression.
                 */
                if (s.match_length <= s.max_lazy_match /*max_insert_length*/ && s.lookahead >= MIN_MATCH$1) {
                    s.match_length--; /* string at strstart already in table */
                    do {
                        s.strstart++;
                        /*** INSERT_STRING(s, s.strstart, hash_head); ***/
                        s.ins_h = ((s.ins_h << s.hash_shift) ^ s.window[s.strstart + MIN_MATCH$1 - 1]) & s.hash_mask;
                        hash_head = s.prev[s.strstart & s.w_mask] = s.head[s.ins_h];
                        s.head[s.ins_h] = s.strstart;
                        /***/
                        /* strstart never exceeds WSIZE-MAX_MATCH, so there are
                         * always MIN_MATCH bytes ahead.
                         */
                    } while (--s.match_length !== 0);
                    s.strstart++;
                }
                else {
                    s.strstart += s.match_length;
                    s.match_length = 0;
                    s.ins_h = s.window[s.strstart];
                    /* UPDATE_HASH(s, s.ins_h, s.window[s.strstart+1]); */
                    s.ins_h = ((s.ins_h << s.hash_shift) ^ s.window[s.strstart + 1]) & s.hash_mask;
                    //#if MIN_MATCH != 3
                    //                Call UPDATE_HASH() MIN_MATCH-3 more times
                    //#endif
                    /* If lookahead < MIN_MATCH, ins_h is garbage, but it does not
                     * matter since it will be recomputed at next deflate call.
                     */
                }
            }
            else {
                /* No match, output a literal byte */
                //Tracevv((stderr,"%c", s.window[s.strstart]));
                /*** _tr_tally_lit(s, s.window[s.strstart], bflush); ***/
                bflush = trees$2._tr_tally(s, 0, s.window[s.strstart]);
                s.lookahead--;
                s.strstart++;
            }
            if (bflush) {
                /*** FLUSH_BLOCK(s, 0); ***/
                flush_block_only(s, false);
                if (s.strm.avail_out === 0) {
                    return BS_NEED_MORE;
                }
                /***/
            }
        }
        s.insert = ((s.strstart < (MIN_MATCH$1 - 1)) ? s.strstart : MIN_MATCH$1 - 1);
        if (flush === Z_FINISH) {
            /*** FLUSH_BLOCK(s, 1); ***/
            flush_block_only(s, true);
            if (s.strm.avail_out === 0) {
                return BS_FINISH_STARTED;
            }
            /***/
            return BS_FINISH_DONE;
        }
        if (s.last_lit) {
            /*** FLUSH_BLOCK(s, 0); ***/
            flush_block_only(s, false);
            if (s.strm.avail_out === 0) {
                return BS_NEED_MORE;
            }
            /***/
        }
        return BS_BLOCK_DONE;
    }
    /* ===========================================================================
     * Same as above, but achieves better compression. We use a lazy
     * evaluation for matches: a match is finally adopted only if there is
     * no better match at the next window position.
     */
    function deflate_slow(s, flush) {
        var hash_head; /* head of hash chain */
        var bflush; /* set if current block must be flushed */
        var max_insert;
        /* Process the input block. */
        for (;;) {
            /* Make sure that we always have enough lookahead, except
             * at the end of the input file. We need MAX_MATCH bytes
             * for the next match, plus MIN_MATCH bytes to insert the
             * string following the next match.
             */
            if (s.lookahead < MIN_LOOKAHEAD) {
                fill_window(s);
                if (s.lookahead < MIN_LOOKAHEAD && flush === Z_NO_FLUSH) {
                    return BS_NEED_MORE;
                }
                if (s.lookahead === 0) {
                    break;
                } /* flush the current block */
            }
            /* Insert the string window[strstart .. strstart+2] in the
             * dictionary, and set hash_head to the head of the hash chain:
             */
            hash_head = 0 /*NIL*/;
            if (s.lookahead >= MIN_MATCH$1) {
                /*** INSERT_STRING(s, s.strstart, hash_head); ***/
                s.ins_h = ((s.ins_h << s.hash_shift) ^ s.window[s.strstart + MIN_MATCH$1 - 1]) & s.hash_mask;
                hash_head = s.prev[s.strstart & s.w_mask] = s.head[s.ins_h];
                s.head[s.ins_h] = s.strstart;
                /***/
            }
            /* Find the longest match, discarding those <= prev_length.
             */
            s.prev_length = s.match_length;
            s.prev_match = s.match_start;
            s.match_length = MIN_MATCH$1 - 1;
            if (hash_head !== 0 /*NIL*/ && s.prev_length < s.max_lazy_match &&
                s.strstart - hash_head <= (s.w_size - MIN_LOOKAHEAD) /*MAX_DIST(s)*/) {
                /* To simplify the code, we prevent matches with the string
                 * of window index 0 (in particular we have to avoid a match
                 * of the string with itself at the start of the input file).
                 */
                s.match_length = longest_match(s, hash_head);
                /* longest_match() sets match_start */
                if (s.match_length <= 5 &&
                    (s.strategy === Z_FILTERED || (s.match_length === MIN_MATCH$1 && s.strstart - s.match_start > 4096 /*TOO_FAR*/))) {
                    /* If prev_match is also MIN_MATCH, match_start is garbage
                     * but we will ignore the current match anyway.
                     */
                    s.match_length = MIN_MATCH$1 - 1;
                }
            }
            /* If there was a match at the previous step and the current
             * match is not better, output the previous match:
             */
            if (s.prev_length >= MIN_MATCH$1 && s.match_length <= s.prev_length) {
                max_insert = s.strstart + s.lookahead - MIN_MATCH$1;
                /* Do not insert strings in hash table beyond this. */
                //check_match(s, s.strstart-1, s.prev_match, s.prev_length);
                /***_tr_tally_dist(s, s.strstart - 1 - s.prev_match,
                               s.prev_length - MIN_MATCH, bflush);***/
                bflush = trees$2._tr_tally(s, s.strstart - 1 - s.prev_match, s.prev_length - MIN_MATCH$1);
                /* Insert in hash table all strings up to the end of the match.
                 * strstart-1 and strstart are already inserted. If there is not
                 * enough lookahead, the last two strings are not inserted in
                 * the hash table.
                 */
                s.lookahead -= s.prev_length - 1;
                s.prev_length -= 2;
                do {
                    if (++s.strstart <= max_insert) {
                        /*** INSERT_STRING(s, s.strstart, hash_head); ***/
                        s.ins_h = ((s.ins_h << s.hash_shift) ^ s.window[s.strstart + MIN_MATCH$1 - 1]) & s.hash_mask;
                        hash_head = s.prev[s.strstart & s.w_mask] = s.head[s.ins_h];
                        s.head[s.ins_h] = s.strstart;
                        /***/
                    }
                } while (--s.prev_length !== 0);
                s.match_available = 0;
                s.match_length = MIN_MATCH$1 - 1;
                s.strstart++;
                if (bflush) {
                    /*** FLUSH_BLOCK(s, 0); ***/
                    flush_block_only(s, false);
                    if (s.strm.avail_out === 0) {
                        return BS_NEED_MORE;
                    }
                    /***/
                }
            }
            else if (s.match_available) {
                /* If there was no match at the previous position, output a
                 * single literal. If there was a match but the current match
                 * is longer, truncate the previous match to a single literal.
                 */
                //Tracevv((stderr,"%c", s->window[s->strstart-1]));
                /*** _tr_tally_lit(s, s.window[s.strstart-1], bflush); ***/
                bflush = trees$2._tr_tally(s, 0, s.window[s.strstart - 1]);
                if (bflush) {
                    /*** FLUSH_BLOCK_ONLY(s, 0) ***/
                    flush_block_only(s, false);
                    /***/
                }
                s.strstart++;
                s.lookahead--;
                if (s.strm.avail_out === 0) {
                    return BS_NEED_MORE;
                }
            }
            else {
                /* There is no previous match to compare with, wait for
                 * the next step to decide.
                 */
                s.match_available = 1;
                s.strstart++;
                s.lookahead--;
            }
        }
        //Assert (flush != Z_NO_FLUSH, "no flush?");
        if (s.match_available) {
            //Tracevv((stderr,"%c", s->window[s->strstart-1]));
            /*** _tr_tally_lit(s, s.window[s.strstart-1], bflush); ***/
            bflush = trees$2._tr_tally(s, 0, s.window[s.strstart - 1]);
            s.match_available = 0;
        }
        s.insert = s.strstart < MIN_MATCH$1 - 1 ? s.strstart : MIN_MATCH$1 - 1;
        if (flush === Z_FINISH) {
            /*** FLUSH_BLOCK(s, 1); ***/
            flush_block_only(s, true);
            if (s.strm.avail_out === 0) {
                return BS_FINISH_STARTED;
            }
            /***/
            return BS_FINISH_DONE;
        }
        if (s.last_lit) {
            /*** FLUSH_BLOCK(s, 0); ***/
            flush_block_only(s, false);
            if (s.strm.avail_out === 0) {
                return BS_NEED_MORE;
            }
            /***/
        }
        return BS_BLOCK_DONE;
    }
    /* ===========================================================================
     * For Z_RLE, simply look for runs of bytes, generate matches only of distance
     * one.  Do not maintain a hash table.  (It will be regenerated if this run of
     * deflate switches away from Z_RLE.)
     */
    function deflate_rle(s, flush) {
        var bflush; /* set if current block must be flushed */
        var prev; /* byte at distance one to match */
        var scan, strend; /* scan goes up to strend for length of run */
        var _win = s.window;
        for (;;) {
            /* Make sure that we always have enough lookahead, except
             * at the end of the input file. We need MAX_MATCH bytes
             * for the longest run, plus one for the unrolled loop.
             */
            if (s.lookahead <= MAX_MATCH$1) {
                fill_window(s);
                if (s.lookahead <= MAX_MATCH$1 && flush === Z_NO_FLUSH) {
                    return BS_NEED_MORE;
                }
                if (s.lookahead === 0) {
                    break;
                } /* flush the current block */
            }
            /* See how many times the previous byte repeats */
            s.match_length = 0;
            if (s.lookahead >= MIN_MATCH$1 && s.strstart > 0) {
                scan = s.strstart - 1;
                prev = _win[scan];
                if (prev === _win[++scan] && prev === _win[++scan] && prev === _win[++scan]) {
                    strend = s.strstart + MAX_MATCH$1;
                    do {
                        /*jshint noempty:false*/
                    } while (prev === _win[++scan] && prev === _win[++scan] &&
                        prev === _win[++scan] && prev === _win[++scan] &&
                        prev === _win[++scan] && prev === _win[++scan] &&
                        prev === _win[++scan] && prev === _win[++scan] &&
                        scan < strend);
                    s.match_length = MAX_MATCH$1 - (strend - scan);
                    if (s.match_length > s.lookahead) {
                        s.match_length = s.lookahead;
                    }
                }
                //Assert(scan <= s->window+(uInt)(s->window_size-1), "wild scan");
            }
            /* Emit match if have run of MIN_MATCH or longer, else emit literal */
            if (s.match_length >= MIN_MATCH$1) {
                //check_match(s, s.strstart, s.strstart - 1, s.match_length);
                /*** _tr_tally_dist(s, 1, s.match_length - MIN_MATCH, bflush); ***/
                bflush = trees$2._tr_tally(s, 1, s.match_length - MIN_MATCH$1);
                s.lookahead -= s.match_length;
                s.strstart += s.match_length;
                s.match_length = 0;
            }
            else {
                /* No match, output a literal byte */
                //Tracevv((stderr,"%c", s->window[s->strstart]));
                /*** _tr_tally_lit(s, s.window[s.strstart], bflush); ***/
                bflush = trees$2._tr_tally(s, 0, s.window[s.strstart]);
                s.lookahead--;
                s.strstart++;
            }
            if (bflush) {
                /*** FLUSH_BLOCK(s, 0); ***/
                flush_block_only(s, false);
                if (s.strm.avail_out === 0) {
                    return BS_NEED_MORE;
                }
                /***/
            }
        }
        s.insert = 0;
        if (flush === Z_FINISH) {
            /*** FLUSH_BLOCK(s, 1); ***/
            flush_block_only(s, true);
            if (s.strm.avail_out === 0) {
                return BS_FINISH_STARTED;
            }
            /***/
            return BS_FINISH_DONE;
        }
        if (s.last_lit) {
            /*** FLUSH_BLOCK(s, 0); ***/
            flush_block_only(s, false);
            if (s.strm.avail_out === 0) {
                return BS_NEED_MORE;
            }
            /***/
        }
        return BS_BLOCK_DONE;
    }
    /* ===========================================================================
     * For Z_HUFFMAN_ONLY, do not look for matches.  Do not maintain a hash table.
     * (It will be regenerated if this run of deflate switches away from Huffman.)
     */
    function deflate_huff(s, flush) {
        var bflush; /* set if current block must be flushed */
        for (;;) {
            /* Make sure that we have a literal to write. */
            if (s.lookahead === 0) {
                fill_window(s);
                if (s.lookahead === 0) {
                    if (flush === Z_NO_FLUSH) {
                        return BS_NEED_MORE;
                    }
                    break; /* flush the current block */
                }
            }
            /* Output a literal byte */
            s.match_length = 0;
            //Tracevv((stderr,"%c", s->window[s->strstart]));
            /*** _tr_tally_lit(s, s.window[s.strstart], bflush); ***/
            bflush = trees$2._tr_tally(s, 0, s.window[s.strstart]);
            s.lookahead--;
            s.strstart++;
            if (bflush) {
                /*** FLUSH_BLOCK(s, 0); ***/
                flush_block_only(s, false);
                if (s.strm.avail_out === 0) {
                    return BS_NEED_MORE;
                }
                /***/
            }
        }
        s.insert = 0;
        if (flush === Z_FINISH) {
            /*** FLUSH_BLOCK(s, 1); ***/
            flush_block_only(s, true);
            if (s.strm.avail_out === 0) {
                return BS_FINISH_STARTED;
            }
            /***/
            return BS_FINISH_DONE;
        }
        if (s.last_lit) {
            /*** FLUSH_BLOCK(s, 0); ***/
            flush_block_only(s, false);
            if (s.strm.avail_out === 0) {
                return BS_NEED_MORE;
            }
            /***/
        }
        return BS_BLOCK_DONE;
    }
    /* Values for max_lazy_match, good_match and max_chain_length, depending on
     * the desired pack level (0..9). The values given below have been tuned to
     * exclude worst case performance for pathological files. Better values may be
     * found for specific files.
     */
    function Config(good_length, max_lazy, nice_length, max_chain, func) {
        this.good_length = good_length;
        this.max_lazy = max_lazy;
        this.nice_length = nice_length;
        this.max_chain = max_chain;
        this.func = func;
    }
    var configuration_table;
    configuration_table = [
        /*      good lazy nice chain */
        new Config(0, 0, 0, 0, deflate_stored),
        new Config(4, 4, 8, 4, deflate_fast),
        new Config(4, 5, 16, 8, deflate_fast),
        new Config(4, 6, 32, 32, deflate_fast),
        new Config(4, 4, 16, 16, deflate_slow),
        new Config(8, 16, 32, 32, deflate_slow),
        new Config(8, 16, 128, 128, deflate_slow),
        new Config(8, 32, 128, 256, deflate_slow),
        new Config(32, 128, 258, 1024, deflate_slow),
        new Config(32, 258, 258, 4096, deflate_slow) /* 9 max compression */
    ];
    /* ===========================================================================
     * Initialize the "longest match" routines for a new zlib stream
     */
    function lm_init(s) {
        s.window_size = 2 * s.w_size;
        /*** CLEAR_HASH(s); ***/
        zero$1(s.head); // Fill with NIL (= 0);
        /* Set the default configuration parameters:
         */
        s.max_lazy_match = configuration_table[s.level].max_lazy;
        s.good_match = configuration_table[s.level].good_length;
        s.nice_match = configuration_table[s.level].nice_length;
        s.max_chain_length = configuration_table[s.level].max_chain;
        s.strstart = 0;
        s.block_start = 0;
        s.lookahead = 0;
        s.insert = 0;
        s.match_length = s.prev_length = MIN_MATCH$1 - 1;
        s.match_available = 0;
        s.ins_h = 0;
    }
    function DeflateState() {
        this.strm = null; /* pointer back to this zlib stream */
        this.status = 0; /* as the name implies */
        this.pending_buf = null; /* output still pending */
        this.pending_buf_size = 0; /* size of pending_buf */
        this.pending_out = 0; /* next pending byte to output to the stream */
        this.pending = 0; /* nb of bytes in the pending buffer */
        this.wrap = 0; /* bit 0 true for zlib, bit 1 true for gzip */
        this.gzhead = null; /* gzip header information to write */
        this.gzindex = 0; /* where in extra, name, or comment */
        this.method = Z_DEFLATED; /* can only be DEFLATED */
        this.last_flush = -1; /* value of flush param for previous deflate call */
        this.w_size = 0; /* LZ77 window size (32K by default) */
        this.w_bits = 0; /* log2(w_size)  (8..16) */
        this.w_mask = 0; /* w_size - 1 */
        this.window = null;
        /* Sliding window. Input bytes are read into the second half of the window,
         * and move to the first half later to keep a dictionary of at least wSize
         * bytes. With this organization, matches are limited to a distance of
         * wSize-MAX_MATCH bytes, but this ensures that IO is always
         * performed with a length multiple of the block size.
         */
        this.window_size = 0;
        /* Actual size of window: 2*wSize, except when the user input buffer
         * is directly used as sliding window.
         */
        this.prev = null;
        /* Link to older string with same hash index. To limit the size of this
         * array to 64K, this link is maintained only for the last 32K strings.
         * An index in this array is thus a window index modulo 32K.
         */
        this.head = null; /* Heads of the hash chains or NIL. */
        this.ins_h = 0; /* hash index of string to be inserted */
        this.hash_size = 0; /* number of elements in hash table */
        this.hash_bits = 0; /* log2(hash_size) */
        this.hash_mask = 0; /* hash_size-1 */
        this.hash_shift = 0;
        /* Number of bits by which ins_h must be shifted at each input
         * step. It must be such that after MIN_MATCH steps, the oldest
         * byte no longer takes part in the hash key, that is:
         *   hash_shift * MIN_MATCH >= hash_bits
         */
        this.block_start = 0;
        /* Window position at the beginning of the current output block. Gets
         * negative when the window is moved backwards.
         */
        this.match_length = 0; /* length of best match */
        this.prev_match = 0; /* previous match */
        this.match_available = 0; /* set if previous match exists */
        this.strstart = 0; /* start of string to insert */
        this.match_start = 0; /* start of matching string */
        this.lookahead = 0; /* number of valid bytes ahead in window */
        this.prev_length = 0;
        /* Length of the best match at previous step. Matches not greater than this
         * are discarded. This is used in the lazy match evaluation.
         */
        this.max_chain_length = 0;
        /* To speed up deflation, hash chains are never searched beyond this
         * length.  A higher limit improves compression ratio but degrades the
         * speed.
         */
        this.max_lazy_match = 0;
        /* Attempt to find a better match only when the current match is strictly
         * smaller than this value. This mechanism is used only for compression
         * levels >= 4.
         */
        // That's alias to max_lazy_match, don't use directly
        //this.max_insert_length = 0;
        /* Insert new strings in the hash table only if the match length is not
         * greater than this length. This saves time but degrades compression.
         * max_insert_length is used only for compression levels <= 3.
         */
        this.level = 0; /* compression level (1..9) */
        this.strategy = 0; /* favor or force Huffman coding*/
        this.good_match = 0;
        /* Use a faster search when the previous match is longer than this */
        this.nice_match = 0; /* Stop searching when current match exceeds this */
        /* used by trees.c: */
        /* Didn't use ct_data typedef below to suppress compiler warning */
        // struct ct_data_s dyn_ltree[HEAP_SIZE];   /* literal and length tree */
        // struct ct_data_s dyn_dtree[2*D_CODES+1]; /* distance tree */
        // struct ct_data_s bl_tree[2*BL_CODES+1];  /* Huffman tree for bit lengths */
        // Use flat array of DOUBLE size, with interleaved fata,
        // because JS does not support effective
        this.dyn_ltree = new utils.Buf16(HEAP_SIZE$1 * 2);
        this.dyn_dtree = new utils.Buf16((2 * D_CODES$1 + 1) * 2);
        this.bl_tree = new utils.Buf16((2 * BL_CODES$1 + 1) * 2);
        zero$1(this.dyn_ltree);
        zero$1(this.dyn_dtree);
        zero$1(this.bl_tree);
        this.l_desc = null; /* desc. for literal tree */
        this.d_desc = null; /* desc. for distance tree */
        this.bl_desc = null; /* desc. for bit length tree */
        //ush bl_count[MAX_BITS+1];
        this.bl_count = new utils.Buf16(MAX_BITS$1 + 1);
        /* number of codes at each bit length for an optimal tree */
        //int heap[2*L_CODES+1];      /* heap used to build the Huffman trees */
        this.heap = new utils.Buf16(2 * L_CODES$1 + 1); /* heap used to build the Huffman trees */
        zero$1(this.heap);
        this.heap_len = 0; /* number of elements in the heap */
        this.heap_max = 0; /* element of largest frequency */
        /* The sons of heap[n] are heap[2*n] and heap[2*n+1]. heap[0] is not used.
         * The same heap array is used to build all trees.
         */
        this.depth = new utils.Buf16(2 * L_CODES$1 + 1); //uch depth[2*L_CODES+1];
        zero$1(this.depth);
        /* Depth of each subtree used as tie breaker for trees of equal frequency
         */
        this.l_buf = 0; /* buffer index for literals or lengths */
        this.lit_bufsize = 0;
        /* Size of match buffer for literals/lengths.  There are 4 reasons for
         * limiting lit_bufsize to 64K:
         *   - frequencies can be kept in 16 bit counters
         *   - if compression is not successful for the first block, all input
         *     data is still in the window so we can still emit a stored block even
         *     when input comes from standard input.  (This can also be done for
         *     all blocks if lit_bufsize is not greater than 32K.)
         *   - if compression is not successful for a file smaller than 64K, we can
         *     even emit a stored file instead of a stored block (saving 5 bytes).
         *     This is applicable only for zip (not gzip or zlib).
         *   - creating new Huffman trees less frequently may not provide fast
         *     adaptation to changes in the input data statistics. (Take for
         *     example a binary file with poorly compressible code followed by
         *     a highly compressible string table.) Smaller buffer sizes give
         *     fast adaptation but have of course the overhead of transmitting
         *     trees more frequently.
         *   - I can't count above 4
         */
        this.last_lit = 0; /* running index in l_buf */
        this.d_buf = 0;
        /* Buffer index for distances. To simplify the code, d_buf and l_buf have
         * the same number of elements. To use different lengths, an extra flag
         * array would be necessary.
         */
        this.opt_len = 0; /* bit length of current block with optimal trees */
        this.static_len = 0; /* bit length of current block with static trees */
        this.matches = 0; /* number of string matches in current block */
        this.insert = 0; /* bytes at end of window left to insert */
        this.bi_buf = 0;
        /* Output buffer. bits are inserted starting at the bottom (least
         * significant bits).
         */
        this.bi_valid = 0;
        /* Number of valid bits in bi_buf.  All bits above the last valid bit
         * are always zero.
         */
        // Used for window memory init. We safely ignore it for JS. That makes
        // sense only for pointers and memory check tools.
        //this.high_water = 0;
        /* High water mark offset in window for initialized bytes -- bytes above
         * this are set to zero in order to avoid memory check warnings when
         * longest match routines access bytes past the input.  This is then
         * updated to the new high water mark.
         */
    }
    function deflateResetKeep(strm) {
        var s;
        if (!strm || !strm.state) {
            return err(strm, Z_STREAM_ERROR);
        }
        strm.total_in = strm.total_out = 0;
        strm.data_type = Z_UNKNOWN$1;
        s = strm.state;
        s.pending = 0;
        s.pending_out = 0;
        if (s.wrap < 0) {
            s.wrap = -s.wrap;
            /* was made negative by deflate(..., Z_FINISH); */
        }
        s.status = (s.wrap ? INIT_STATE : BUSY_STATE);
        strm.adler = (s.wrap === 2) ?
            0 // crc32(0, Z_NULL, 0)
            :
                1; // adler32(0, Z_NULL, 0)
        s.last_flush = Z_NO_FLUSH;
        trees$2._tr_init(s);
        return Z_OK;
    }
    function deflateReset(strm) {
        var ret = deflateResetKeep(strm);
        if (ret === Z_OK) {
            lm_init(strm.state);
        }
        return ret;
    }
    function deflateSetHeader(strm, head) {
        if (!strm || !strm.state) {
            return Z_STREAM_ERROR;
        }
        if (strm.state.wrap !== 2) {
            return Z_STREAM_ERROR;
        }
        strm.state.gzhead = head;
        return Z_OK;
    }
    function deflateInit2(strm, level, method, windowBits, memLevel, strategy) {
        if (!strm) { // === Z_NULL
            return Z_STREAM_ERROR;
        }
        var wrap = 1;
        if (level === Z_DEFAULT_COMPRESSION) {
            level = 6;
        }
        if (windowBits < 0) { /* suppress zlib wrapper */
            wrap = 0;
            windowBits = -windowBits;
        }
        else if (windowBits > 15) {
            wrap = 2; /* write gzip wrapper instead */
            windowBits -= 16;
        }
        if (memLevel < 1 || memLevel > MAX_MEM_LEVEL || method !== Z_DEFLATED ||
            windowBits < 8 || windowBits > 15 || level < 0 || level > 9 ||
            strategy < 0 || strategy > Z_FIXED$1) {
            return err(strm, Z_STREAM_ERROR);
        }
        if (windowBits === 8) {
            windowBits = 9;
        }
        /* until 256-byte window bug fixed */
        var s = new DeflateState();
        strm.state = s;
        s.strm = strm;
        s.wrap = wrap;
        s.gzhead = null;
        s.w_bits = windowBits;
        s.w_size = 1 << s.w_bits;
        s.w_mask = s.w_size - 1;
        s.hash_bits = memLevel + 7;
        s.hash_size = 1 << s.hash_bits;
        s.hash_mask = s.hash_size - 1;
        s.hash_shift = ~~((s.hash_bits + MIN_MATCH$1 - 1) / MIN_MATCH$1);
        s.window = new utils.Buf8(s.w_size * 2);
        s.head = new utils.Buf16(s.hash_size);
        s.prev = new utils.Buf16(s.w_size);
        // Don't need mem init magic for JS.
        //s.high_water = 0;  /* nothing written to s->window yet */
        s.lit_bufsize = 1 << (memLevel + 6); /* 16K elements by default */
        s.pending_buf_size = s.lit_bufsize * 4;
        //overlay = (ushf *) ZALLOC(strm, s->lit_bufsize, sizeof(ush)+2);
        //s->pending_buf = (uchf *) overlay;
        s.pending_buf = new utils.Buf8(s.pending_buf_size);
        // It is offset from `s.pending_buf` (size is `s.lit_bufsize * 2`)
        //s->d_buf = overlay + s->lit_bufsize/sizeof(ush);
        s.d_buf = 1 * s.lit_bufsize;
        //s->l_buf = s->pending_buf + (1+sizeof(ush))*s->lit_bufsize;
        s.l_buf = (1 + 2) * s.lit_bufsize;
        s.level = level;
        s.strategy = strategy;
        s.method = method;
        return deflateReset(strm);
    }
    function deflateInit(strm, level) {
        return deflateInit2(strm, level, Z_DEFLATED, MAX_WBITS, DEF_MEM_LEVEL, Z_DEFAULT_STRATEGY);
    }
    function deflate(strm, flush) {
        var old_flush, s;
        var beg, val; // for gzip header write only
        if (!strm || !strm.state ||
            flush > Z_BLOCK || flush < 0) {
            return strm ? err(strm, Z_STREAM_ERROR) : Z_STREAM_ERROR;
        }
        s = strm.state;
        if (!strm.output ||
            (!strm.input && strm.avail_in !== 0) ||
            (s.status === FINISH_STATE && flush !== Z_FINISH)) {
            return err(strm, (strm.avail_out === 0) ? Z_BUF_ERROR : Z_STREAM_ERROR);
        }
        s.strm = strm; /* just in case */
        old_flush = s.last_flush;
        s.last_flush = flush;
        /* Write the header */
        if (s.status === INIT_STATE) {
            if (s.wrap === 2) { // GZIP header
                strm.adler = 0; //crc32(0L, Z_NULL, 0);
                put_byte(s, 31);
                put_byte(s, 139);
                put_byte(s, 8);
                if (!s.gzhead) { // s->gzhead == Z_NULL
                    put_byte(s, 0);
                    put_byte(s, 0);
                    put_byte(s, 0);
                    put_byte(s, 0);
                    put_byte(s, 0);
                    put_byte(s, s.level === 9 ? 2 :
                        (s.strategy >= Z_HUFFMAN_ONLY || s.level < 2 ?
                            4 : 0));
                    put_byte(s, OS_CODE);
                    s.status = BUSY_STATE;
                }
                else {
                    put_byte(s, (s.gzhead.text ? 1 : 0) +
                        (s.gzhead.hcrc ? 2 : 0) +
                        (!s.gzhead.extra ? 0 : 4) +
                        (!s.gzhead.name ? 0 : 8) +
                        (!s.gzhead.comment ? 0 : 16));
                    put_byte(s, s.gzhead.time & 0xff);
                    put_byte(s, (s.gzhead.time >> 8) & 0xff);
                    put_byte(s, (s.gzhead.time >> 16) & 0xff);
                    put_byte(s, (s.gzhead.time >> 24) & 0xff);
                    put_byte(s, s.level === 9 ? 2 :
                        (s.strategy >= Z_HUFFMAN_ONLY || s.level < 2 ?
                            4 : 0));
                    put_byte(s, s.gzhead.os & 0xff);
                    if (s.gzhead.extra && s.gzhead.extra.length) {
                        put_byte(s, s.gzhead.extra.length & 0xff);
                        put_byte(s, (s.gzhead.extra.length >> 8) & 0xff);
                    }
                    if (s.gzhead.hcrc) {
                        strm.adler = crc32$2(strm.adler, s.pending_buf, s.pending, 0);
                    }
                    s.gzindex = 0;
                    s.status = EXTRA_STATE;
                }
            }
            else // DEFLATE header
             {
                var header = (Z_DEFLATED + ((s.w_bits - 8) << 4)) << 8;
                var level_flags = -1;
                if (s.strategy >= Z_HUFFMAN_ONLY || s.level < 2) {
                    level_flags = 0;
                }
                else if (s.level < 6) {
                    level_flags = 1;
                }
                else if (s.level === 6) {
                    level_flags = 2;
                }
                else {
                    level_flags = 3;
                }
                header |= (level_flags << 6);
                if (s.strstart !== 0) {
                    header |= PRESET_DICT;
                }
                header += 31 - (header % 31);
                s.status = BUSY_STATE;
                putShortMSB(s, header);
                /* Save the adler32 of the preset dictionary: */
                if (s.strstart !== 0) {
                    putShortMSB(s, strm.adler >>> 16);
                    putShortMSB(s, strm.adler & 0xffff);
                }
                strm.adler = 1; // adler32(0L, Z_NULL, 0);
            }
        }
        //#ifdef GZIP
        if (s.status === EXTRA_STATE) {
            if (s.gzhead.extra /* != Z_NULL*/) {
                beg = s.pending; /* start of bytes to update crc */
                while (s.gzindex < (s.gzhead.extra.length & 0xffff)) {
                    if (s.pending === s.pending_buf_size) {
                        if (s.gzhead.hcrc && s.pending > beg) {
                            strm.adler = crc32$2(strm.adler, s.pending_buf, s.pending - beg, beg);
                        }
                        flush_pending(strm);
                        beg = s.pending;
                        if (s.pending === s.pending_buf_size) {
                            break;
                        }
                    }
                    put_byte(s, s.gzhead.extra[s.gzindex] & 0xff);
                    s.gzindex++;
                }
                if (s.gzhead.hcrc && s.pending > beg) {
                    strm.adler = crc32$2(strm.adler, s.pending_buf, s.pending - beg, beg);
                }
                if (s.gzindex === s.gzhead.extra.length) {
                    s.gzindex = 0;
                    s.status = NAME_STATE;
                }
            }
            else {
                s.status = NAME_STATE;
            }
        }
        if (s.status === NAME_STATE) {
            if (s.gzhead.name /* != Z_NULL*/) {
                beg = s.pending; /* start of bytes to update crc */
                //int val;
                do {
                    if (s.pending === s.pending_buf_size) {
                        if (s.gzhead.hcrc && s.pending > beg) {
                            strm.adler = crc32$2(strm.adler, s.pending_buf, s.pending - beg, beg);
                        }
                        flush_pending(strm);
                        beg = s.pending;
                        if (s.pending === s.pending_buf_size) {
                            val = 1;
                            break;
                        }
                    }
                    // JS specific: little magic to add zero terminator to end of string
                    if (s.gzindex < s.gzhead.name.length) {
                        val = s.gzhead.name.charCodeAt(s.gzindex++) & 0xff;
                    }
                    else {
                        val = 0;
                    }
                    put_byte(s, val);
                } while (val !== 0);
                if (s.gzhead.hcrc && s.pending > beg) {
                    strm.adler = crc32$2(strm.adler, s.pending_buf, s.pending - beg, beg);
                }
                if (val === 0) {
                    s.gzindex = 0;
                    s.status = COMMENT_STATE;
                }
            }
            else {
                s.status = COMMENT_STATE;
            }
        }
        if (s.status === COMMENT_STATE) {
            if (s.gzhead.comment /* != Z_NULL*/) {
                beg = s.pending; /* start of bytes to update crc */
                //int val;
                do {
                    if (s.pending === s.pending_buf_size) {
                        if (s.gzhead.hcrc && s.pending > beg) {
                            strm.adler = crc32$2(strm.adler, s.pending_buf, s.pending - beg, beg);
                        }
                        flush_pending(strm);
                        beg = s.pending;
                        if (s.pending === s.pending_buf_size) {
                            val = 1;
                            break;
                        }
                    }
                    // JS specific: little magic to add zero terminator to end of string
                    if (s.gzindex < s.gzhead.comment.length) {
                        val = s.gzhead.comment.charCodeAt(s.gzindex++) & 0xff;
                    }
                    else {
                        val = 0;
                    }
                    put_byte(s, val);
                } while (val !== 0);
                if (s.gzhead.hcrc && s.pending > beg) {
                    strm.adler = crc32$2(strm.adler, s.pending_buf, s.pending - beg, beg);
                }
                if (val === 0) {
                    s.status = HCRC_STATE;
                }
            }
            else {
                s.status = HCRC_STATE;
            }
        }
        if (s.status === HCRC_STATE) {
            if (s.gzhead.hcrc) {
                if (s.pending + 2 > s.pending_buf_size) {
                    flush_pending(strm);
                }
                if (s.pending + 2 <= s.pending_buf_size) {
                    put_byte(s, strm.adler & 0xff);
                    put_byte(s, (strm.adler >> 8) & 0xff);
                    strm.adler = 0; //crc32(0L, Z_NULL, 0);
                    s.status = BUSY_STATE;
                }
            }
            else {
                s.status = BUSY_STATE;
            }
        }
        //#endif
        /* Flush as much pending output as possible */
        if (s.pending !== 0) {
            flush_pending(strm);
            if (strm.avail_out === 0) {
                /* Since avail_out is 0, deflate will be called again with
                 * more output space, but possibly with both pending and
                 * avail_in equal to zero. There won't be anything to do,
                 * but this is not an error situation so make sure we
                 * return OK instead of BUF_ERROR at next call of deflate:
                 */
                s.last_flush = -1;
                return Z_OK;
            }
            /* Make sure there is something to do and avoid duplicate consecutive
             * flushes. For repeated and useless calls with Z_FINISH, we keep
             * returning Z_STREAM_END instead of Z_BUF_ERROR.
             */
        }
        else if (strm.avail_in === 0 && rank(flush) <= rank(old_flush) &&
            flush !== Z_FINISH) {
            return err(strm, Z_BUF_ERROR);
        }
        /* User must not provide more input after the first FINISH: */
        if (s.status === FINISH_STATE && strm.avail_in !== 0) {
            return err(strm, Z_BUF_ERROR);
        }
        /* Start a new block or continue the current one.
         */
        if (strm.avail_in !== 0 || s.lookahead !== 0 ||
            (flush !== Z_NO_FLUSH && s.status !== FINISH_STATE)) {
            var bstate = (s.strategy === Z_HUFFMAN_ONLY) ? deflate_huff(s, flush) :
                (s.strategy === Z_RLE ? deflate_rle(s, flush) :
                    configuration_table[s.level].func(s, flush));
            if (bstate === BS_FINISH_STARTED || bstate === BS_FINISH_DONE) {
                s.status = FINISH_STATE;
            }
            if (bstate === BS_NEED_MORE || bstate === BS_FINISH_STARTED) {
                if (strm.avail_out === 0) {
                    s.last_flush = -1;
                    /* avoid BUF_ERROR next call, see above */
                }
                return Z_OK;
                /* If flush != Z_NO_FLUSH && avail_out == 0, the next call
                 * of deflate should use the same flush parameter to make sure
                 * that the flush is complete. So we don't have to output an
                 * empty block here, this will be done at next call. This also
                 * ensures that for a very small output buffer, we emit at most
                 * one empty block.
                 */
            }
            if (bstate === BS_BLOCK_DONE) {
                if (flush === Z_PARTIAL_FLUSH) {
                    trees$2._tr_align(s);
                }
                else if (flush !== Z_BLOCK) { /* FULL_FLUSH or SYNC_FLUSH */
                    trees$2._tr_stored_block(s, 0, 0, false);
                    /* For a full flush, this empty block will be recognized
                     * as a special marker by inflate_sync().
                     */
                    if (flush === Z_FULL_FLUSH) {
                        /*** CLEAR_HASH(s); ***/ /* forget history */
                        zero$1(s.head); // Fill with NIL (= 0);
                        if (s.lookahead === 0) {
                            s.strstart = 0;
                            s.block_start = 0;
                            s.insert = 0;
                        }
                    }
                }
                flush_pending(strm);
                if (strm.avail_out === 0) {
                    s.last_flush = -1; /* avoid BUF_ERROR at next call, see above */
                    return Z_OK;
                }
            }
        }
        //Assert(strm->avail_out > 0, "bug2");
        //if (strm.avail_out <= 0) { throw new Error("bug2");}
        if (flush !== Z_FINISH) {
            return Z_OK;
        }
        if (s.wrap <= 0) {
            return Z_STREAM_END;
        }
        /* Write the trailer */
        if (s.wrap === 2) {
            put_byte(s, strm.adler & 0xff);
            put_byte(s, (strm.adler >> 8) & 0xff);
            put_byte(s, (strm.adler >> 16) & 0xff);
            put_byte(s, (strm.adler >> 24) & 0xff);
            put_byte(s, strm.total_in & 0xff);
            put_byte(s, (strm.total_in >> 8) & 0xff);
            put_byte(s, (strm.total_in >> 16) & 0xff);
            put_byte(s, (strm.total_in >> 24) & 0xff);
        }
        else {
            putShortMSB(s, strm.adler >>> 16);
            putShortMSB(s, strm.adler & 0xffff);
        }
        flush_pending(strm);
        /* If avail_out is zero, the application will call deflate again
         * to flush the rest.
         */
        if (s.wrap > 0) {
            s.wrap = -s.wrap;
        }
        /* write the trailer only once! */
        return s.pending !== 0 ? Z_OK : Z_STREAM_END;
    }
    function deflateEnd(strm) {
        var status;
        if (!strm /*== Z_NULL*/ || !strm.state /*== Z_NULL*/) {
            return Z_STREAM_ERROR;
        }
        status = strm.state.status;
        if (status !== INIT_STATE &&
            status !== EXTRA_STATE &&
            status !== NAME_STATE &&
            status !== COMMENT_STATE &&
            status !== HCRC_STATE &&
            status !== BUSY_STATE &&
            status !== FINISH_STATE) {
            return err(strm, Z_STREAM_ERROR);
        }
        strm.state = null;
        return status === BUSY_STATE ? err(strm, Z_DATA_ERROR) : Z_OK;
    }
    /* =========================================================================
     * Initializes the compression dictionary from the given byte
     * sequence without producing any compressed output.
     */
    function deflateSetDictionary(strm, dictionary) {
        var dictLength = dictionary.length;
        var s;
        var str, n;
        var wrap;
        var avail;
        var next;
        var input;
        var tmpDict;
        if (!strm /*== Z_NULL*/ || !strm.state /*== Z_NULL*/) {
            return Z_STREAM_ERROR;
        }
        s = strm.state;
        wrap = s.wrap;
        if (wrap === 2 || (wrap === 1 && s.status !== INIT_STATE) || s.lookahead) {
            return Z_STREAM_ERROR;
        }
        /* when using zlib wrappers, compute Adler-32 for provided dictionary */
        if (wrap === 1) {
            /* adler32(strm->adler, dictionary, dictLength); */
            strm.adler = adler32$2(strm.adler, dictionary, dictLength, 0);
        }
        s.wrap = 0; /* avoid computing Adler-32 in read_buf */
        /* if dictionary would fill window, just replace the history */
        if (dictLength >= s.w_size) {
            if (wrap === 0) { /* already empty otherwise */
                /*** CLEAR_HASH(s); ***/
                zero$1(s.head); // Fill with NIL (= 0);
                s.strstart = 0;
                s.block_start = 0;
                s.insert = 0;
            }
            /* use the tail */
            // dictionary = dictionary.slice(dictLength - s.w_size);
            tmpDict = new utils.Buf8(s.w_size);
            utils.arraySet(tmpDict, dictionary, dictLength - s.w_size, s.w_size, 0);
            dictionary = tmpDict;
            dictLength = s.w_size;
        }
        /* insert dictionary into window and hash */
        avail = strm.avail_in;
        next = strm.next_in;
        input = strm.input;
        strm.avail_in = dictLength;
        strm.next_in = 0;
        strm.input = dictionary;
        fill_window(s);
        while (s.lookahead >= MIN_MATCH$1) {
            str = s.strstart;
            n = s.lookahead - (MIN_MATCH$1 - 1);
            do {
                /* UPDATE_HASH(s, s->ins_h, s->window[str + MIN_MATCH-1]); */
                s.ins_h = ((s.ins_h << s.hash_shift) ^ s.window[str + MIN_MATCH$1 - 1]) & s.hash_mask;
                s.prev[str & s.w_mask] = s.head[s.ins_h];
                s.head[s.ins_h] = str;
                str++;
            } while (--n);
            s.strstart = str;
            s.lookahead = MIN_MATCH$1 - 1;
            fill_window(s);
        }
        s.strstart += s.lookahead;
        s.block_start = s.strstart;
        s.insert = s.lookahead;
        s.lookahead = 0;
        s.match_length = s.prev_length = MIN_MATCH$1 - 1;
        s.match_available = 0;
        strm.next_in = next;
        strm.input = input;
        strm.avail_in = avail;
        s.wrap = wrap;
        return Z_OK;
    }
    var deflateInit_1 = deflateInit;
    var deflateInit2_1 = deflateInit2;
    var deflateReset_1 = deflateReset;
    var deflateResetKeep_1 = deflateResetKeep;
    var deflateSetHeader_1 = deflateSetHeader;
    var deflate_2 = deflate;
    var deflateEnd_1 = deflateEnd;
    var deflateSetDictionary_1 = deflateSetDictionary;
    var deflateInfo = 'pako deflate (from Nodeca project)';
    /* Not implemented
    exports.deflateBound = deflateBound;
    exports.deflateCopy = deflateCopy;
    exports.deflateParams = deflateParams;
    exports.deflatePending = deflatePending;
    exports.deflatePrime = deflatePrime;
    exports.deflateTune = deflateTune;
    */
    var deflate_1 = {
        deflateInit: deflateInit_1,
        deflateInit2: deflateInit2_1,
        deflateReset: deflateReset_1,
        deflateResetKeep: deflateResetKeep_1,
        deflateSetHeader: deflateSetHeader_1,
        deflate: deflate_2,
        deflateEnd: deflateEnd_1,
        deflateSetDictionary: deflateSetDictionary_1,
        deflateInfo: deflateInfo
    };

    var deflate$1 = /*#__PURE__*/Object.freeze({
        default: deflate_1,
        __moduleExports: deflate_1,
        deflateInit: deflateInit_1,
        deflateInit2: deflateInit2_1,
        deflateReset: deflateReset_1,
        deflateResetKeep: deflateResetKeep_1,
        deflateSetHeader: deflateSetHeader_1,
        deflate: deflate_2,
        deflateEnd: deflateEnd_1,
        deflateSetDictionary: deflateSetDictionary_1,
        deflateInfo: deflateInfo
    });

    // Quick check if we can use fast array to bin string conversion
    //
    // - apply(Array) can fail on Android 2.2
    // - apply(Uint8Array) can fail on iOS 5.1 Safari
    //
    var STR_APPLY_OK = true;
    var STR_APPLY_UIA_OK = true;
    try {
        String.fromCharCode.apply(null, [0]);
    }
    catch (__) {
        STR_APPLY_OK = false;
    }
    try {
        String.fromCharCode.apply(null, new Uint8Array(1));
    }
    catch (__) {
        STR_APPLY_UIA_OK = false;
    }
    // Table with utf8 lengths (calculated by first byte of sequence)
    // Note, that 5 & 6-byte values and some 4-byte values can not be represented in JS,
    // because max possible codepoint is 0x10ffff
    var _utf8len = new utils.Buf8(256);
    for (var q = 0; q < 256; q++) {
        _utf8len[q] = (q >= 252 ? 6 : q >= 248 ? 5 : q >= 240 ? 4 : q >= 224 ? 3 : q >= 192 ? 2 : 1);
    }
    _utf8len[254] = _utf8len[254] = 1; // Invalid sequence start
    // convert string to array (typed, when possible)
    var string2buf = function (str) {
        var buf, c, c2, m_pos, i, str_len = str.length, buf_len = 0;
        // count binary size
        for (m_pos = 0; m_pos < str_len; m_pos++) {
            c = str.charCodeAt(m_pos);
            if ((c & 0xfc00) === 0xd800 && (m_pos + 1 < str_len)) {
                c2 = str.charCodeAt(m_pos + 1);
                if ((c2 & 0xfc00) === 0xdc00) {
                    c = 0x10000 + ((c - 0xd800) << 10) + (c2 - 0xdc00);
                    m_pos++;
                }
            }
            buf_len += c < 0x80 ? 1 : c < 0x800 ? 2 : c < 0x10000 ? 3 : 4;
        }
        // allocate buffer
        buf = new utils.Buf8(buf_len);
        // convert
        for (i = 0, m_pos = 0; i < buf_len; m_pos++) {
            c = str.charCodeAt(m_pos);
            if ((c & 0xfc00) === 0xd800 && (m_pos + 1 < str_len)) {
                c2 = str.charCodeAt(m_pos + 1);
                if ((c2 & 0xfc00) === 0xdc00) {
                    c = 0x10000 + ((c - 0xd800) << 10) + (c2 - 0xdc00);
                    m_pos++;
                }
            }
            if (c < 0x80) {
                /* one byte */
                buf[i++] = c;
            }
            else if (c < 0x800) {
                /* two bytes */
                buf[i++] = 0xC0 | (c >>> 6);
                buf[i++] = 0x80 | (c & 0x3f);
            }
            else if (c < 0x10000) {
                /* three bytes */
                buf[i++] = 0xE0 | (c >>> 12);
                buf[i++] = 0x80 | (c >>> 6 & 0x3f);
                buf[i++] = 0x80 | (c & 0x3f);
            }
            else {
                /* four bytes */
                buf[i++] = 0xf0 | (c >>> 18);
                buf[i++] = 0x80 | (c >>> 12 & 0x3f);
                buf[i++] = 0x80 | (c >>> 6 & 0x3f);
                buf[i++] = 0x80 | (c & 0x3f);
            }
        }
        return buf;
    };
    // Helper (used in 2 places)
    function buf2binstring(buf, len) {
        // On Chrome, the arguments in a function call that are allowed is `65534`.
        // If the length of the buffer is smaller than that, we can use this optimization,
        // otherwise we will take a slower path.
        if (len < 65534) {
            if ((buf.subarray && STR_APPLY_UIA_OK) || (!buf.subarray && STR_APPLY_OK)) {
                return String.fromCharCode.apply(null, utils.shrinkBuf(buf, len));
            }
        }
        var result = '';
        for (var i = 0; i < len; i++) {
            result += String.fromCharCode(buf[i]);
        }
        return result;
    }
    // Convert byte array to binary string
    var buf2binstring_1 = function (buf) {
        return buf2binstring(buf, buf.length);
    };
    // Convert binary string (typed, when possible)
    var binstring2buf = function (str) {
        var buf = new utils.Buf8(str.length);
        for (var i = 0, len = buf.length; i < len; i++) {
            buf[i] = str.charCodeAt(i);
        }
        return buf;
    };
    // convert array to string
    var buf2string = function (buf, max) {
        var i, out, c, c_len;
        var len = max || buf.length;
        // Reserve max possible length (2 words per char)
        // NB: by unknown reasons, Array is significantly faster for
        //     String.fromCharCode.apply than Uint16Array.
        var utf16buf = new Array(len * 2);
        for (out = 0, i = 0; i < len;) {
            c = buf[i++];
            // quick process ascii
            if (c < 0x80) {
                utf16buf[out++] = c;
                continue;
            }
            c_len = _utf8len[c];
            // skip 5 & 6 byte codes
            if (c_len > 4) {
                utf16buf[out++] = 0xfffd;
                i += c_len - 1;
                continue;
            }
            // apply mask on first byte
            c &= c_len === 2 ? 0x1f : c_len === 3 ? 0x0f : 0x07;
            // join the rest
            while (c_len > 1 && i < len) {
                c = (c << 6) | (buf[i++] & 0x3f);
                c_len--;
            }
            // terminated by end of string?
            if (c_len > 1) {
                utf16buf[out++] = 0xfffd;
                continue;
            }
            if (c < 0x10000) {
                utf16buf[out++] = c;
            }
            else {
                c -= 0x10000;
                utf16buf[out++] = 0xd800 | ((c >> 10) & 0x3ff);
                utf16buf[out++] = 0xdc00 | (c & 0x3ff);
            }
        }
        return buf2binstring(utf16buf, out);
    };
    // Calculate max possible position in utf8 buffer,
    // that will not break sequence. If that's not possible
    // - (very small limits) return max size as is.
    //
    // buf[] - utf8 bytes array
    // max   - length limit (mandatory);
    var utf8border = function (buf, max) {
        var pos;
        max = max || buf.length;
        if (max > buf.length) {
            max = buf.length;
        }
        // go back from last position, until start of sequence found
        pos = max - 1;
        while (pos >= 0 && (buf[pos] & 0xC0) === 0x80) {
            pos--;
        }
        // Very small and broken sequence,
        // return max, because we should return something anyway.
        if (pos < 0) {
            return max;
        }
        // If we came to start of buffer - that means buffer is too small,
        // return max too.
        if (pos === 0) {
            return max;
        }
        return (pos + _utf8len[buf[pos]] > max) ? pos : max;
    };
    var strings = {
        string2buf: string2buf,
        buf2binstring: buf2binstring_1,
        binstring2buf: binstring2buf,
        buf2string: buf2string,
        utf8border: utf8border
    };

    var strings$1 = /*#__PURE__*/Object.freeze({
        default: strings,
        __moduleExports: strings,
        string2buf: string2buf,
        buf2binstring: buf2binstring_1,
        binstring2buf: binstring2buf,
        buf2string: buf2string,
        utf8border: utf8border
    });

    // (C) 1995-2013 Jean-loup Gailly and Mark Adler
    // (C) 2014-2017 Vitaly Puzrin and Andrey Tupitsin
    //
    // This software is provided 'as-is', without any express or implied
    // warranty. In no event will the authors be held liable for any damages
    // arising from the use of this software.
    //
    // Permission is granted to anyone to use this software for any purpose,
    // including commercial applications, and to alter it and redistribute it
    // freely, subject to the following restrictions:
    //
    // 1. The origin of this software must not be misrepresented; you must not
    //   claim that you wrote the original software. If you use this software
    //   in a product, an acknowledgment in the product documentation would be
    //   appreciated but is not required.
    // 2. Altered source versions must be plainly marked as such, and must not be
    //   misrepresented as being the original software.
    // 3. This notice may not be removed or altered from any source distribution.
    function ZStream() {
        /* next input byte */
        this.input = null; // JS specific, because we have no pointers
        this.next_in = 0;
        /* number of bytes available at input */
        this.avail_in = 0;
        /* total number of input bytes read so far */
        this.total_in = 0;
        /* next output byte should be put there */
        this.output = null; // JS specific, because we have no pointers
        this.next_out = 0;
        /* remaining free space at output */
        this.avail_out = 0;
        /* total number of bytes output so far */
        this.total_out = 0;
        /* last error message, NULL if no error */
        this.msg = '' /*Z_NULL*/;
        /* not visible by applications */
        this.state = null;
        /* best guess about the data type: binary or text */
        this.data_type = 2 /*Z_UNKNOWN*/;
        /* adler32 value of the uncompressed data */
        this.adler = 0;
    }
    var zstream = ZStream;

    var zstream$1 = /*#__PURE__*/Object.freeze({
        default: zstream,
        __moduleExports: zstream
    });

    var zlib_deflate = (deflate$1 && deflate_1) || deflate$1;

    var strings$2 = (strings$1 && strings) || strings$1;

    var ZStream$1 = (zstream$1 && zstream) || zstream$1;

    var toString = Object.prototype.toString;
    /* Public constants ==========================================================*/
    /* ===========================================================================*/
    var Z_NO_FLUSH$1 = 0;
    var Z_FINISH$1 = 4;
    var Z_OK$1 = 0;
    var Z_STREAM_END$1 = 1;
    var Z_SYNC_FLUSH = 2;
    var Z_DEFAULT_COMPRESSION$1 = -1;
    var Z_DEFAULT_STRATEGY$1 = 0;
    var Z_DEFLATED$1 = 8;
    /* ===========================================================================*/
    /**
     * class Deflate
     *
     * Generic JS-style wrapper for zlib calls. If you don't need
     * streaming behaviour - use more simple functions: [[deflate]],
     * [[deflateRaw]] and [[gzip]].
     **/
    /* internal
     * Deflate.chunks -> Array
     *
     * Chunks of output data, if [[Deflate#onData]] not overridden.
     **/
    /**
     * Deflate.result -> Uint8Array|Array
     *
     * Compressed result, generated by default [[Deflate#onData]]
     * and [[Deflate#onEnd]] handlers. Filled after you push last chunk
     * (call [[Deflate#push]] with `Z_FINISH` / `true` param)  or if you
     * push a chunk with explicit flush (call [[Deflate#push]] with
     * `Z_SYNC_FLUSH` param).
     **/
    /**
     * Deflate.err -> Number
     *
     * Error code after deflate finished. 0 (Z_OK) on success.
     * You will not need it in real life, because deflate errors
     * are possible only on wrong options or bad `onData` / `onEnd`
     * custom handlers.
     **/
    /**
     * Deflate.msg -> String
     *
     * Error message, if [[Deflate.err]] != 0
     **/
    /**
     * new Deflate(options)
     * - options (Object): zlib deflate options.
     *
     * Creates new deflator instance with specified params. Throws exception
     * on bad params. Supported options:
     *
     * - `level`
     * - `windowBits`
     * - `memLevel`
     * - `strategy`
     * - `dictionary`
     *
     * [http://zlib.net/manual.html#Advanced](http://zlib.net/manual.html#Advanced)
     * for more information on these.
     *
     * Additional options, for internal needs:
     *
     * - `chunkSize` - size of generated data chunks (16K by default)
     * - `raw` (Boolean) - do raw deflate
     * - `gzip` (Boolean) - create gzip wrapper
     * - `to` (String) - if equal to 'string', then result will be "binary string"
     *    (each char code [0..255])
     * - `header` (Object) - custom header for gzip
     *   - `text` (Boolean) - true if compressed data believed to be text
     *   - `time` (Number) - modification time, unix timestamp
     *   - `os` (Number) - operation system code
     *   - `extra` (Array) - array of bytes with extra data (max 65536)
     *   - `name` (String) - file name (binary string)
     *   - `comment` (String) - comment (binary string)
     *   - `hcrc` (Boolean) - true if header crc should be added
     *
     * ##### Example:
     *
     * ```javascript
     * var pako = require('pako')
     *   , chunk1 = Uint8Array([1,2,3,4,5,6,7,8,9])
     *   , chunk2 = Uint8Array([10,11,12,13,14,15,16,17,18,19]);
     *
     * var deflate = new pako.Deflate({ level: 3});
     *
     * deflate.push(chunk1, false);
     * deflate.push(chunk2, true);  // true -> last chunk
     *
     * if (deflate.err) { throw new Error(deflate.err); }
     *
     * console.log(deflate.result);
     * ```
     **/
    function Deflate(options) {
        if (!(this instanceof Deflate))
            return new Deflate(options);
        this.options = utils.assign({
            level: Z_DEFAULT_COMPRESSION$1,
            method: Z_DEFLATED$1,
            chunkSize: 16384,
            windowBits: 15,
            memLevel: 8,
            strategy: Z_DEFAULT_STRATEGY$1,
            to: ''
        }, options || {});
        var opt = this.options;
        if (opt.raw && (opt.windowBits > 0)) {
            opt.windowBits = -opt.windowBits;
        }
        else if (opt.gzip && (opt.windowBits > 0) && (opt.windowBits < 16)) {
            opt.windowBits += 16;
        }
        this.err = 0; // error code, if happens (0 = Z_OK)
        this.msg = ''; // error message
        this.ended = false; // used to avoid multiple onEnd() calls
        this.chunks = []; // chunks of compressed data
        this.strm = new ZStream$1();
        this.strm.avail_out = 0;
        var status = zlib_deflate.deflateInit2(this.strm, opt.level, opt.method, opt.windowBits, opt.memLevel, opt.strategy);
        if (status !== Z_OK$1) {
            throw new Error(msg[status]);
        }
        if (opt.header) {
            zlib_deflate.deflateSetHeader(this.strm, opt.header);
        }
        if (opt.dictionary) {
            var dict;
            // Convert data if needed
            if (typeof opt.dictionary === 'string') {
                // If we need to compress text, change encoding to utf8.
                dict = strings$2.string2buf(opt.dictionary);
            }
            else if (toString.call(opt.dictionary) === '[object ArrayBuffer]') {
                dict = new Uint8Array(opt.dictionary);
            }
            else {
                dict = opt.dictionary;
            }
            status = zlib_deflate.deflateSetDictionary(this.strm, dict);
            if (status !== Z_OK$1) {
                throw new Error(msg[status]);
            }
            this._dict_set = true;
        }
    }
    /**
     * Deflate#push(data[, mode]) -> Boolean
     * - data (Uint8Array|Array|ArrayBuffer|String): input data. Strings will be
     *   converted to utf8 byte sequence.
     * - mode (Number|Boolean): 0..6 for corresponding Z_NO_FLUSH..Z_TREE modes.
     *   See constants. Skipped or `false` means Z_NO_FLUSH, `true` means Z_FINISH.
     *
     * Sends input data to deflate pipe, generating [[Deflate#onData]] calls with
     * new compressed chunks. Returns `true` on success. The last data block must have
     * mode Z_FINISH (or `true`). That will flush internal pending buffers and call
     * [[Deflate#onEnd]]. For interim explicit flushes (without ending the stream) you
     * can use mode Z_SYNC_FLUSH, keeping the compression context.
     *
     * On fail call [[Deflate#onEnd]] with error code and return false.
     *
     * We strongly recommend to use `Uint8Array` on input for best speed (output
     * array format is detected automatically). Also, don't skip last param and always
     * use the same type in your code (boolean or number). That will improve JS speed.
     *
     * For regular `Array`-s make sure all elements are [0..255].
     *
     * ##### Example
     *
     * ```javascript
     * push(chunk, false); // push one of data chunks
     * ...
     * push(chunk, true);  // push last chunk
     * ```
     **/
    Deflate.prototype.push = function (data, mode) {
        var strm = this.strm;
        var chunkSize = this.options.chunkSize;
        var status, _mode;
        if (this.ended) {
            return false;
        }
        _mode = (mode === ~~mode) ? mode : ((mode === true) ? Z_FINISH$1 : Z_NO_FLUSH$1);
        // Convert data if needed
        if (typeof data === 'string') {
            // If we need to compress text, change encoding to utf8.
            strm.input = strings$2.string2buf(data);
        }
        else if (toString.call(data) === '[object ArrayBuffer]') {
            strm.input = new Uint8Array(data);
        }
        else {
            strm.input = data;
        }
        strm.next_in = 0;
        strm.avail_in = strm.input.length;
        do {
            if (strm.avail_out === 0) {
                strm.output = new utils.Buf8(chunkSize);
                strm.next_out = 0;
                strm.avail_out = chunkSize;
            }
            status = zlib_deflate.deflate(strm, _mode); /* no bad return value */
            if (status !== Z_STREAM_END$1 && status !== Z_OK$1) {
                this.onEnd(status);
                this.ended = true;
                return false;
            }
            if (strm.avail_out === 0 || (strm.avail_in === 0 && (_mode === Z_FINISH$1 || _mode === Z_SYNC_FLUSH))) {
                if (this.options.to === 'string') {
                    this.onData(strings$2.buf2binstring(utils.shrinkBuf(strm.output, strm.next_out)));
                }
                else {
                    this.onData(utils.shrinkBuf(strm.output, strm.next_out));
                }
            }
        } while ((strm.avail_in > 0 || strm.avail_out === 0) && status !== Z_STREAM_END$1);
        // Finalize on the last chunk.
        if (_mode === Z_FINISH$1) {
            status = zlib_deflate.deflateEnd(this.strm);
            this.onEnd(status);
            this.ended = true;
            return status === Z_OK$1;
        }
        // callback interim results if Z_SYNC_FLUSH.
        if (_mode === Z_SYNC_FLUSH) {
            this.onEnd(Z_OK$1);
            strm.avail_out = 0;
            return true;
        }
        return true;
    };
    /**
     * Deflate#onData(chunk) -> Void
     * - chunk (Uint8Array|Array|String): output data. Type of array depends
     *   on js engine support. When string output requested, each chunk
     *   will be string.
     *
     * By default, stores data blocks in `chunks[]` property and glue
     * those in `onEnd`. Override this handler, if you need another behaviour.
     **/
    Deflate.prototype.onData = function (chunk) {
        this.chunks.push(chunk);
    };
    /**
     * Deflate#onEnd(status) -> Void
     * - status (Number): deflate status. 0 (Z_OK) on success,
     *   other if not.
     *
     * Called once after you tell deflate that the input stream is
     * complete (Z_FINISH) or should be flushed (Z_SYNC_FLUSH)
     * or if an error happened. By default - join collected chunks,
     * free memory and fill `results` / `err` properties.
     **/
    Deflate.prototype.onEnd = function (status) {
        // On success - join
        if (status === Z_OK$1) {
            if (this.options.to === 'string') {
                this.result = this.chunks.join('');
            }
            else {
                this.result = utils.flattenChunks(this.chunks);
            }
        }
        this.chunks = [];
        this.err = status;
        this.msg = this.strm.msg;
    };
    /**
     * deflate(data[, options]) -> Uint8Array|Array|String
     * - data (Uint8Array|Array|String): input data to compress.
     * - options (Object): zlib deflate options.
     *
     * Compress `data` with deflate algorithm and `options`.
     *
     * Supported options are:
     *
     * - level
     * - windowBits
     * - memLevel
     * - strategy
     * - dictionary
     *
     * [http://zlib.net/manual.html#Advanced](http://zlib.net/manual.html#Advanced)
     * for more information on these.
     *
     * Sugar (options):
     *
     * - `raw` (Boolean) - say that we work with raw stream, if you don't wish to specify
     *   negative windowBits implicitly.
     * - `to` (String) - if equal to 'string', then result will be "binary string"
     *    (each char code [0..255])
     *
     * ##### Example:
     *
     * ```javascript
     * var pako = require('pako')
     *   , data = Uint8Array([1,2,3,4,5,6,7,8,9]);
     *
     * console.log(pako.deflate(data));
     * ```
     **/
    function deflate$2(input, options) {
        var deflator = new Deflate(options);
        deflator.push(input, true);
        // That will never happens, if you don't cheat with options :)
        if (deflator.err) {
            throw deflator.msg || msg[deflator.err];
        }
        return deflator.result;
    }
    /**
     * deflateRaw(data[, options]) -> Uint8Array|Array|String
     * - data (Uint8Array|Array|String): input data to compress.
     * - options (Object): zlib deflate options.
     *
     * The same as [[deflate]], but creates raw data, without wrapper
     * (header and adler32 crc).
     **/
    function deflateRaw(input, options) {
        options = options || {};
        options.raw = true;
        return deflate$2(input, options);
    }
    /**
     * gzip(data[, options]) -> Uint8Array|Array|String
     * - data (Uint8Array|Array|String): input data to compress.
     * - options (Object): zlib deflate options.
     *
     * The same as [[deflate]], but create gzip wrapper instead of
     * deflate one.
     **/
    function gzip(input, options) {
        options = options || {};
        options.gzip = true;
        return deflate$2(input, options);
    }
    var Deflate_1 = Deflate;
    var deflate_2$1 = deflate$2;
    var deflateRaw_1 = deflateRaw;
    var gzip_1 = gzip;
    var deflate_1$1 = {
        Deflate: Deflate_1,
        deflate: deflate_2$1,
        deflateRaw: deflateRaw_1,
        gzip: gzip_1
    };

    var deflate$3 = /*#__PURE__*/Object.freeze({
        default: deflate_1$1,
        __moduleExports: deflate_1$1,
        Deflate: Deflate_1,
        deflate: deflate_2$1,
        deflateRaw: deflateRaw_1,
        gzip: gzip_1
    });

    // (C) 1995-2013 Jean-loup Gailly and Mark Adler
    // (C) 2014-2017 Vitaly Puzrin and Andrey Tupitsin
    //
    // This software is provided 'as-is', without any express or implied
    // warranty. In no event will the authors be held liable for any damages
    // arising from the use of this software.
    //
    // Permission is granted to anyone to use this software for any purpose,
    // including commercial applications, and to alter it and redistribute it
    // freely, subject to the following restrictions:
    //
    // 1. The origin of this software must not be misrepresented; you must not
    //   claim that you wrote the original software. If you use this software
    //   in a product, an acknowledgment in the product documentation would be
    //   appreciated but is not required.
    // 2. Altered source versions must be plainly marked as such, and must not be
    //   misrepresented as being the original software.
    // 3. This notice may not be removed or altered from any source distribution.
    // See state defs from inflate.js
    var BAD = 30; /* got a data error -- remain here until reset */
    var TYPE = 12; /* i: waiting for type bits, including last-flag bit */
    /*
       Decode literal, length, and distance codes and write out the resulting
       literal and match bytes until either not enough input or output is
       available, an end-of-block is encountered, or a data error is encountered.
       When large enough input and output buffers are supplied to inflate(), for
       example, a 16K input buffer and a 64K output buffer, more than 95% of the
       inflate execution time is spent in this routine.

       Entry assumptions:

            state.mode === LEN
            strm.avail_in >= 6
            strm.avail_out >= 258
            start >= strm.avail_out
            state.bits < 8

       On return, state.mode is one of:

            LEN -- ran out of enough output space or enough available input
            TYPE -- reached end of block code, inflate() to interpret next block
            BAD -- error in block data

       Notes:

        - The maximum input bits used by a length/distance pair is 15 bits for the
          length code, 5 bits for the length extra, 15 bits for the distance code,
          and 13 bits for the distance extra.  This totals 48 bits, or six bytes.
          Therefore if strm.avail_in >= 6, then there is enough input to avoid
          checking for available input while decoding.

        - The maximum bytes that a single length/distance pair can output is 258
          bytes, which is the maximum length that can be coded.  inflate_fast()
          requires strm.avail_out >= 258 for each loop to avoid checking for
          output space.
     */
    var inffast = function inflate_fast(strm, start) {
        var state;
        var _in; /* local strm.input */
        var last; /* have enough input while in < last */
        var _out; /* local strm.output */
        var beg; /* inflate()'s initial strm.output */
        var end; /* while out < end, enough space available */
        //#ifdef INFLATE_STRICT
        var dmax; /* maximum distance from zlib header */
        //#endif
        var wsize; /* window size or zero if not using window */
        var whave; /* valid bytes in the window */
        var wnext; /* window write index */
        // Use `s_window` instead `window`, avoid conflict with instrumentation tools
        var s_window; /* allocated sliding window, if wsize != 0 */
        var hold; /* local strm.hold */
        var bits; /* local strm.bits */
        var lcode; /* local strm.lencode */
        var dcode; /* local strm.distcode */
        var lmask; /* mask for first level of length codes */
        var dmask; /* mask for first level of distance codes */
        var here; /* retrieved table entry */
        var op; /* code bits, operation, extra bits, or */
        /*  window position, window bytes to copy */
        var len; /* match length, unused bytes */
        var dist; /* match distance */
        var from; /* where to copy match from */
        var from_source;
        var input, output; // JS specific, because we have no pointers
        /* copy state to local variables */
        state = strm.state;
        //here = state.here;
        _in = strm.next_in;
        input = strm.input;
        last = _in + (strm.avail_in - 5);
        _out = strm.next_out;
        output = strm.output;
        beg = _out - (start - strm.avail_out);
        end = _out + (strm.avail_out - 257);
        //#ifdef INFLATE_STRICT
        dmax = state.dmax;
        //#endif
        wsize = state.wsize;
        whave = state.whave;
        wnext = state.wnext;
        s_window = state.window;
        hold = state.hold;
        bits = state.bits;
        lcode = state.lencode;
        dcode = state.distcode;
        lmask = (1 << state.lenbits) - 1;
        dmask = (1 << state.distbits) - 1;
        /* decode literals and length/distances until end-of-block or not enough
           input data or output space */
        top: do {
            if (bits < 15) {
                hold += input[_in++] << bits;
                bits += 8;
                hold += input[_in++] << bits;
                bits += 8;
            }
            here = lcode[hold & lmask];
            dolen: for (;;) { // Goto emulation
                op = here >>> 24 /*here.bits*/;
                hold >>>= op;
                bits -= op;
                op = (here >>> 16) & 0xff /*here.op*/;
                if (op === 0) { /* literal */
                    //Tracevv((stderr, here.val >= 0x20 && here.val < 0x7f ?
                    //        "inflate:         literal '%c'\n" :
                    //        "inflate:         literal 0x%02x\n", here.val));
                    output[_out++] = here & 0xffff /*here.val*/;
                }
                else if (op & 16) { /* length base */
                    len = here & 0xffff /*here.val*/;
                    op &= 15; /* number of extra bits */
                    if (op) {
                        if (bits < op) {
                            hold += input[_in++] << bits;
                            bits += 8;
                        }
                        len += hold & ((1 << op) - 1);
                        hold >>>= op;
                        bits -= op;
                    }
                    //Tracevv((stderr, "inflate:         length %u\n", len));
                    if (bits < 15) {
                        hold += input[_in++] << bits;
                        bits += 8;
                        hold += input[_in++] << bits;
                        bits += 8;
                    }
                    here = dcode[hold & dmask];
                    dodist: for (;;) { // goto emulation
                        op = here >>> 24 /*here.bits*/;
                        hold >>>= op;
                        bits -= op;
                        op = (here >>> 16) & 0xff /*here.op*/;
                        if (op & 16) { /* distance base */
                            dist = here & 0xffff /*here.val*/;
                            op &= 15; /* number of extra bits */
                            if (bits < op) {
                                hold += input[_in++] << bits;
                                bits += 8;
                                if (bits < op) {
                                    hold += input[_in++] << bits;
                                    bits += 8;
                                }
                            }
                            dist += hold & ((1 << op) - 1);
                            //#ifdef INFLATE_STRICT
                            if (dist > dmax) {
                                strm.msg = 'invalid distance too far back';
                                state.mode = BAD;
                                break top;
                            }
                            //#endif
                            hold >>>= op;
                            bits -= op;
                            //Tracevv((stderr, "inflate:         distance %u\n", dist));
                            op = _out - beg; /* max distance in output */
                            if (dist > op) { /* see if copy from window */
                                op = dist - op; /* distance back in window */
                                if (op > whave) {
                                    if (state.sane) {
                                        strm.msg = 'invalid distance too far back';
                                        state.mode = BAD;
                                        break top;
                                    }
                                    // (!) This block is disabled in zlib defaults,
                                    // don't enable it for binary compatibility
                                    //#ifdef INFLATE_ALLOW_INVALID_DISTANCE_TOOFAR_ARRR
                                    //                if (len <= op - whave) {
                                    //                  do {
                                    //                    output[_out++] = 0;
                                    //                  } while (--len);
                                    //                  continue top;
                                    //                }
                                    //                len -= op - whave;
                                    //                do {
                                    //                  output[_out++] = 0;
                                    //                } while (--op > whave);
                                    //                if (op === 0) {
                                    //                  from = _out - dist;
                                    //                  do {
                                    //                    output[_out++] = output[from++];
                                    //                  } while (--len);
                                    //                  continue top;
                                    //                }
                                    //#endif
                                }
                                from = 0; // window index
                                from_source = s_window;
                                if (wnext === 0) { /* very common case */
                                    from += wsize - op;
                                    if (op < len) { /* some from window */
                                        len -= op;
                                        do {
                                            output[_out++] = s_window[from++];
                                        } while (--op);
                                        from = _out - dist; /* rest from output */
                                        from_source = output;
                                    }
                                }
                                else if (wnext < op) { /* wrap around window */
                                    from += wsize + wnext - op;
                                    op -= wnext;
                                    if (op < len) { /* some from end of window */
                                        len -= op;
                                        do {
                                            output[_out++] = s_window[from++];
                                        } while (--op);
                                        from = 0;
                                        if (wnext < len) { /* some from start of window */
                                            op = wnext;
                                            len -= op;
                                            do {
                                                output[_out++] = s_window[from++];
                                            } while (--op);
                                            from = _out - dist; /* rest from output */
                                            from_source = output;
                                        }
                                    }
                                }
                                else { /* contiguous in window */
                                    from += wnext - op;
                                    if (op < len) { /* some from window */
                                        len -= op;
                                        do {
                                            output[_out++] = s_window[from++];
                                        } while (--op);
                                        from = _out - dist; /* rest from output */
                                        from_source = output;
                                    }
                                }
                                while (len > 2) {
                                    output[_out++] = from_source[from++];
                                    output[_out++] = from_source[from++];
                                    output[_out++] = from_source[from++];
                                    len -= 3;
                                }
                                if (len) {
                                    output[_out++] = from_source[from++];
                                    if (len > 1) {
                                        output[_out++] = from_source[from++];
                                    }
                                }
                            }
                            else {
                                from = _out - dist; /* copy direct from output */
                                do { /* minimum length is three */
                                    output[_out++] = output[from++];
                                    output[_out++] = output[from++];
                                    output[_out++] = output[from++];
                                    len -= 3;
                                } while (len > 2);
                                if (len) {
                                    output[_out++] = output[from++];
                                    if (len > 1) {
                                        output[_out++] = output[from++];
                                    }
                                }
                            }
                        }
                        else if ((op & 64) === 0) { /* 2nd level distance code */
                            here = dcode[(here & 0xffff) /*here.val*/ + (hold & ((1 << op) - 1))];
                            continue dodist;
                        }
                        else {
                            strm.msg = 'invalid distance code';
                            state.mode = BAD;
                            break top;
                        }
                        break; // need to emulate goto via "continue"
                    }
                }
                else if ((op & 64) === 0) { /* 2nd level length code */
                    here = lcode[(here & 0xffff) /*here.val*/ + (hold & ((1 << op) - 1))];
                    continue dolen;
                }
                else if (op & 32) { /* end-of-block */
                    //Tracevv((stderr, "inflate:         end of block\n"));
                    state.mode = TYPE;
                    break top;
                }
                else {
                    strm.msg = 'invalid literal/length code';
                    state.mode = BAD;
                    break top;
                }
                break; // need to emulate goto via "continue"
            }
        } while (_in < last && _out < end);
        /* return unused bytes (on entry, bits < 8, so in won't go too far back) */
        len = bits >> 3;
        _in -= len;
        bits -= len << 3;
        hold &= (1 << bits) - 1;
        /* update state and return */
        strm.next_in = _in;
        strm.next_out = _out;
        strm.avail_in = (_in < last ? 5 + (last - _in) : 5 - (_in - last));
        strm.avail_out = (_out < end ? 257 + (end - _out) : 257 - (_out - end));
        state.hold = hold;
        state.bits = bits;
        return;
    };

    var inffast$1 = /*#__PURE__*/Object.freeze({
        default: inffast,
        __moduleExports: inffast
    });

    // (C) 1995-2013 Jean-loup Gailly and Mark Adler
    // (C) 2014-2017 Vitaly Puzrin and Andrey Tupitsin
    //
    // This software is provided 'as-is', without any express or implied
    // warranty. In no event will the authors be held liable for any damages
    // arising from the use of this software.
    //
    // Permission is granted to anyone to use this software for any purpose,
    // including commercial applications, and to alter it and redistribute it
    // freely, subject to the following restrictions:
    //
    // 1. The origin of this software must not be misrepresented; you must not
    //   claim that you wrote the original software. If you use this software
    //   in a product, an acknowledgment in the product documentation would be
    //   appreciated but is not required.
    // 2. Altered source versions must be plainly marked as such, and must not be
    //   misrepresented as being the original software.
    // 3. This notice may not be removed or altered from any source distribution.
    var MAXBITS = 15;
    var ENOUGH_LENS = 852;
    var ENOUGH_DISTS = 592;
    //var ENOUGH = (ENOUGH_LENS+ENOUGH_DISTS);
    var CODES = 0;
    var LENS = 1;
    var DISTS = 2;
    var lbase = [
        3, 4, 5, 6, 7, 8, 9, 10, 11, 13, 15, 17, 19, 23, 27, 31,
        35, 43, 51, 59, 67, 83, 99, 115, 131, 163, 195, 227, 258, 0, 0
    ];
    var lext = [
        16, 16, 16, 16, 16, 16, 16, 16, 17, 17, 17, 17, 18, 18, 18, 18,
        19, 19, 19, 19, 20, 20, 20, 20, 21, 21, 21, 21, 16, 72, 78
    ];
    var dbase = [
        1, 2, 3, 4, 5, 7, 9, 13, 17, 25, 33, 49, 65, 97, 129, 193,
        257, 385, 513, 769, 1025, 1537, 2049, 3073, 4097, 6145,
        8193, 12289, 16385, 24577, 0, 0
    ];
    var dext = [
        16, 16, 16, 16, 17, 17, 18, 18, 19, 19, 20, 20, 21, 21, 22, 22,
        23, 23, 24, 24, 25, 25, 26, 26, 27, 27,
        28, 28, 29, 29, 64, 64
    ];
    var inftrees = function inflate_table(type, lens, lens_index, codes, table, table_index, work, opts) {
        var bits = opts.bits;
        //here = opts.here; /* table entry for duplication */
        var len = 0; /* a code's length in bits */
        var sym = 0; /* index of code symbols */
        var min = 0, max = 0; /* minimum and maximum code lengths */
        var root = 0; /* number of index bits for root table */
        var curr = 0; /* number of index bits for current table */
        var drop = 0; /* code bits to drop for sub-table */
        var left = 0; /* number of prefix codes available */
        var used = 0; /* code entries in table used */
        var huff = 0; /* Huffman code */
        var incr; /* for incrementing code, index */
        var fill; /* index for replicating entries */
        var low; /* low bits for current root entry */
        var mask; /* mask for low root bits */
        var next; /* next available space in table */
        var base = null; /* base value table to use */
        var base_index = 0;
        //  var shoextra;    /* extra bits table to use */
        var end; /* use base and extra for symbol > end */
        var count = new utils.Buf16(MAXBITS + 1); //[MAXBITS+1];    /* number of codes of each length */
        var offs = new utils.Buf16(MAXBITS + 1); //[MAXBITS+1];     /* offsets in table for each length */
        var extra = null;
        var extra_index = 0;
        var here_bits, here_op, here_val;
        /*
         Process a set of code lengths to create a canonical Huffman code.  The
         code lengths are lens[0..codes-1].  Each length corresponds to the
         symbols 0..codes-1.  The Huffman code is generated by first sorting the
         symbols by length from short to long, and retaining the symbol order
         for codes with equal lengths.  Then the code starts with all zero bits
         for the first code of the shortest length, and the codes are integer
         increments for the same length, and zeros are appended as the length
         increases.  For the deflate format, these bits are stored backwards
         from their more natural integer increment ordering, and so when the
         decoding tables are built in the large loop below, the integer codes
         are incremented backwards.
      
         This routine assumes, but does not check, that all of the entries in
         lens[] are in the range 0..MAXBITS.  The caller must assure this.
         1..MAXBITS is interpreted as that code length.  zero means that that
         symbol does not occur in this code.
      
         The codes are sorted by computing a count of codes for each length,
         creating from that a table of starting indices for each length in the
         sorted table, and then entering the symbols in order in the sorted
         table.  The sorted table is work[], with that space being provided by
         the caller.
      
         The length counts are used for other purposes as well, i.e. finding
         the minimum and maximum length codes, determining if there are any
         codes at all, checking for a valid set of lengths, and looking ahead
         at length counts to determine sub-table sizes when building the
         decoding tables.
         */
        /* accumulate lengths for codes (assumes lens[] all in 0..MAXBITS) */
        for (len = 0; len <= MAXBITS; len++) {
            count[len] = 0;
        }
        for (sym = 0; sym < codes; sym++) {
            count[lens[lens_index + sym]]++;
        }
        /* bound code lengths, force root to be within code lengths */
        root = bits;
        for (max = MAXBITS; max >= 1; max--) {
            if (count[max] !== 0) {
                break;
            }
        }
        if (root > max) {
            root = max;
        }
        if (max === 0) { /* no symbols to code at all */
            //table.op[opts.table_index] = 64;  //here.op = (var char)64;    /* invalid code marker */
            //table.bits[opts.table_index] = 1;   //here.bits = (var char)1;
            //table.val[opts.table_index++] = 0;   //here.val = (var short)0;
            table[table_index++] = (1 << 24) | (64 << 16) | 0;
            //table.op[opts.table_index] = 64;
            //table.bits[opts.table_index] = 1;
            //table.val[opts.table_index++] = 0;
            table[table_index++] = (1 << 24) | (64 << 16) | 0;
            opts.bits = 1;
            return 0; /* no symbols, but wait for decoding to report error */
        }
        for (min = 1; min < max; min++) {
            if (count[min] !== 0) {
                break;
            }
        }
        if (root < min) {
            root = min;
        }
        /* check for an over-subscribed or incomplete set of lengths */
        left = 1;
        for (len = 1; len <= MAXBITS; len++) {
            left <<= 1;
            left -= count[len];
            if (left < 0) {
                return -1;
            } /* over-subscribed */
        }
        if (left > 0 && (type === CODES || max !== 1)) {
            return -1; /* incomplete set */
        }
        /* generate offsets into symbol table for each length for sorting */
        offs[1] = 0;
        for (len = 1; len < MAXBITS; len++) {
            offs[len + 1] = offs[len] + count[len];
        }
        /* sort symbols by length, by symbol order within each length */
        for (sym = 0; sym < codes; sym++) {
            if (lens[lens_index + sym] !== 0) {
                work[offs[lens[lens_index + sym]]++] = sym;
            }
        }
        /*
         Create and fill in decoding tables.  In this loop, the table being
         filled is at next and has curr index bits.  The code being used is huff
         with length len.  That code is converted to an index by dropping drop
         bits off of the bottom.  For codes where len is less than drop + curr,
         those top drop + curr - len bits are incremented through all values to
         fill the table with replicated entries.
      
         root is the number of index bits for the root table.  When len exceeds
         root, sub-tables are created pointed to by the root entry with an index
         of the low root bits of huff.  This is saved in low to check for when a
         new sub-table should be started.  drop is zero when the root table is
         being filled, and drop is root when sub-tables are being filled.
      
         When a new sub-table is needed, it is necessary to look ahead in the
         code lengths to determine what size sub-table is needed.  The length
         counts are used for this, and so count[] is decremented as codes are
         entered in the tables.
      
         used keeps track of how many table entries have been allocated from the
         provided *table space.  It is checked for LENS and DIST tables against
         the constants ENOUGH_LENS and ENOUGH_DISTS to guard against changes in
         the initial root table size constants.  See the comments in inftrees.h
         for more information.
      
         sym increments through all symbols, and the loop terminates when
         all codes of length max, i.e. all codes, have been processed.  This
         routine permits incomplete codes, so another loop after this one fills
         in the rest of the decoding tables with invalid code markers.
         */
        /* set up for code type */
        // poor man optimization - use if-else instead of switch,
        // to avoid deopts in old v8
        if (type === CODES) {
            base = extra = work; /* dummy value--not used */
            end = 19;
        }
        else if (type === LENS) {
            base = lbase;
            base_index -= 257;
            extra = lext;
            extra_index -= 257;
            end = 256;
        }
        else { /* DISTS */
            base = dbase;
            extra = dext;
            end = -1;
        }
        /* initialize opts for loop */
        huff = 0; /* starting code */
        sym = 0; /* starting code symbol */
        len = min; /* starting code length */
        next = table_index; /* current table to fill in */
        curr = root; /* current table index bits */
        drop = 0; /* current bits to drop from code for index */
        low = -1; /* trigger new sub-table when len > root */
        used = 1 << root; /* use root table entries */
        mask = used - 1; /* mask for comparing low */
        /* check available table space */
        if ((type === LENS && used > ENOUGH_LENS) ||
            (type === DISTS && used > ENOUGH_DISTS)) {
            return 1;
        }
        /* process all codes and make table entries */
        for (;;) {
            /* create table entry */
            here_bits = len - drop;
            if (work[sym] < end) {
                here_op = 0;
                here_val = work[sym];
            }
            else if (work[sym] > end) {
                here_op = extra[extra_index + work[sym]];
                here_val = base[base_index + work[sym]];
            }
            else {
                here_op = 32 + 64; /* end of block */
                here_val = 0;
            }
            /* replicate for those indices with low len bits equal to huff */
            incr = 1 << (len - drop);
            fill = 1 << curr;
            min = fill; /* save offset to next table */
            do {
                fill -= incr;
                table[next + (huff >> drop) + fill] = (here_bits << 24) | (here_op << 16) | here_val | 0;
            } while (fill !== 0);
            /* backwards increment the len-bit code huff */
            incr = 1 << (len - 1);
            while (huff & incr) {
                incr >>= 1;
            }
            if (incr !== 0) {
                huff &= incr - 1;
                huff += incr;
            }
            else {
                huff = 0;
            }
            /* go to next symbol, update count, len */
            sym++;
            if (--count[len] === 0) {
                if (len === max) {
                    break;
                }
                len = lens[lens_index + work[sym]];
            }
            /* create new sub-table if needed */
            if (len > root && (huff & mask) !== low) {
                /* if first time, transition to sub-tables */
                if (drop === 0) {
                    drop = root;
                }
                /* increment past last table */
                next += min; /* here min is 1 << curr */
                /* determine length of next table */
                curr = len - drop;
                left = 1 << curr;
                while (curr + drop < max) {
                    left -= count[curr + drop];
                    if (left <= 0) {
                        break;
                    }
                    curr++;
                    left <<= 1;
                }
                /* check for enough space */
                used += 1 << curr;
                if ((type === LENS && used > ENOUGH_LENS) ||
                    (type === DISTS && used > ENOUGH_DISTS)) {
                    return 1;
                }
                /* point entry in root table to sub-table */
                low = huff & mask;
                /*table.op[low] = curr;
                table.bits[low] = root;
                table.val[low] = next - opts.table_index;*/
                table[low] = (root << 24) | (curr << 16) | (next - table_index) | 0;
            }
        }
        /* fill in remaining table entry if code is incomplete (guaranteed to have
         at most one remaining entry, since if the code is incomplete, the
         maximum code length that was allowed to get this far is one bit) */
        if (huff !== 0) {
            //table.op[next + huff] = 64;            /* invalid code marker */
            //table.bits[next + huff] = len - drop;
            //table.val[next + huff] = 0;
            table[next + huff] = ((len - drop) << 24) | (64 << 16) | 0;
        }
        /* set return parameters */
        //opts.table_index += used;
        opts.bits = root;
        return 0;
    };

    var inftrees$1 = /*#__PURE__*/Object.freeze({
        default: inftrees,
        __moduleExports: inftrees
    });

    var inflate_fast = (inffast$1 && inffast) || inffast$1;

    var inflate_table = (inftrees$1 && inftrees) || inftrees$1;

    // (C) 1995-2013 Jean-loup Gailly and Mark Adler
    // (C) 2014-2017 Vitaly Puzrin and Andrey Tupitsin
    //
    // This software is provided 'as-is', without any express or implied
    // warranty. In no event will the authors be held liable for any damages
    // arising from the use of this software.
    //
    // Permission is granted to anyone to use this software for any purpose,
    // including commercial applications, and to alter it and redistribute it
    // freely, subject to the following restrictions:
    //
    // 1. The origin of this software must not be misrepresented; you must not
    //   claim that you wrote the original software. If you use this software
    //   in a product, an acknowledgment in the product documentation would be
    //   appreciated but is not required.
    // 2. Altered source versions must be plainly marked as such, and must not be
    //   misrepresented as being the original software.
    // 3. This notice may not be removed or altered from any source distribution.
    var CODES$1 = 0;
    var LENS$1 = 1;
    var DISTS$1 = 2;
    /* Public constants ==========================================================*/
    /* ===========================================================================*/
    /* Allowed flush values; see deflate() and inflate() below for details */
    //var Z_NO_FLUSH      = 0;
    //var Z_PARTIAL_FLUSH = 1;
    //var Z_SYNC_FLUSH    = 2;
    //var Z_FULL_FLUSH    = 3;
    var Z_FINISH$2 = 4;
    var Z_BLOCK$1 = 5;
    var Z_TREES = 6;
    /* Return codes for the compression/decompression functions. Negative values
     * are errors, positive values are used for special but normal events.
     */
    var Z_OK$2 = 0;
    var Z_STREAM_END$2 = 1;
    var Z_NEED_DICT = 2;
    //var Z_ERRNO         = -1;
    var Z_STREAM_ERROR$1 = -2;
    var Z_DATA_ERROR$1 = -3;
    var Z_MEM_ERROR = -4;
    var Z_BUF_ERROR$1 = -5;
    //var Z_VERSION_ERROR = -6;
    /* The deflate compression method */
    var Z_DEFLATED$2 = 8;
    /* STATES ====================================================================*/
    /* ===========================================================================*/
    var HEAD = 1; /* i: waiting for magic header */
    var FLAGS = 2; /* i: waiting for method and flags (gzip) */
    var TIME = 3; /* i: waiting for modification time (gzip) */
    var OS = 4; /* i: waiting for extra flags and operating system (gzip) */
    var EXLEN = 5; /* i: waiting for extra length (gzip) */
    var EXTRA = 6; /* i: waiting for extra bytes (gzip) */
    var NAME = 7; /* i: waiting for end of file name (gzip) */
    var COMMENT = 8; /* i: waiting for end of comment (gzip) */
    var HCRC = 9; /* i: waiting for header crc (gzip) */
    var DICTID = 10; /* i: waiting for dictionary check value */
    var DICT = 11; /* waiting for inflateSetDictionary() call */
    var TYPE$1 = 12; /* i: waiting for type bits, including last-flag bit */
    var TYPEDO = 13; /* i: same, but skip check to exit inflate on new block */
    var STORED = 14; /* i: waiting for stored size (length and complement) */
    var COPY_ = 15; /* i/o: same as COPY below, but only first time in */
    var COPY = 16; /* i/o: waiting for input or output to copy stored block */
    var TABLE = 17; /* i: waiting for dynamic block table lengths */
    var LENLENS = 18; /* i: waiting for code length code lengths */
    var CODELENS = 19; /* i: waiting for length/lit and distance code lengths */
    var LEN_ = 20; /* i: same as LEN below, but only first time in */
    var LEN = 21; /* i: waiting for length/lit/eob code */
    var LENEXT = 22; /* i: waiting for length extra bits */
    var DIST = 23; /* i: waiting for distance code */
    var DISTEXT = 24; /* i: waiting for distance extra bits */
    var MATCH = 25; /* o: waiting for output space to copy string */
    var LIT = 26; /* o: waiting for output space to write literal */
    var CHECK = 27; /* i: waiting for 32-bit check value */
    var LENGTH = 28; /* i: waiting for 32-bit length (gzip) */
    var DONE = 29; /* finished check, done -- remain here until reset */
    var BAD$1 = 30; /* got a data error -- remain here until reset */
    var MEM = 31; /* got an inflate() memory error -- remain here until reset */
    var SYNC = 32; /* looking for synchronization bytes to restart inflate() */
    /* ===========================================================================*/
    var ENOUGH_LENS$1 = 852;
    var ENOUGH_DISTS$1 = 592;
    //var ENOUGH =  (ENOUGH_LENS+ENOUGH_DISTS);
    var MAX_WBITS$1 = 15;
    /* 32K LZ77 window */
    var DEF_WBITS = MAX_WBITS$1;
    function zswap32(q) {
        return (((q >>> 24) & 0xff) +
            ((q >>> 8) & 0xff00) +
            ((q & 0xff00) << 8) +
            ((q & 0xff) << 24));
    }
    function InflateState() {
        this.mode = 0; /* current inflate mode */
        this.last = false; /* true if processing last block */
        this.wrap = 0; /* bit 0 true for zlib, bit 1 true for gzip */
        this.havedict = false; /* true if dictionary provided */
        this.flags = 0; /* gzip header method and flags (0 if zlib) */
        this.dmax = 0; /* zlib header max distance (INFLATE_STRICT) */
        this.check = 0; /* protected copy of check value */
        this.total = 0; /* protected copy of output count */
        // TODO: may be {}
        this.head = null; /* where to save gzip header information */
        /* sliding window */
        this.wbits = 0; /* log base 2 of requested window size */
        this.wsize = 0; /* window size or zero if not using window */
        this.whave = 0; /* valid bytes in the window */
        this.wnext = 0; /* window write index */
        this.window = null; /* allocated sliding window, if needed */
        /* bit accumulator */
        this.hold = 0; /* input bit accumulator */
        this.bits = 0; /* number of bits in "in" */
        /* for string and stored block copying */
        this.length = 0; /* literal or length of data to copy */
        this.offset = 0; /* distance back to copy string from */
        /* for table and code decoding */
        this.extra = 0; /* extra bits needed */
        /* fixed and dynamic code tables */
        this.lencode = null; /* starting table for length/literal codes */
        this.distcode = null; /* starting table for distance codes */
        this.lenbits = 0; /* index bits for lencode */
        this.distbits = 0; /* index bits for distcode */
        /* dynamic table building */
        this.ncode = 0; /* number of code length code lengths */
        this.nlen = 0; /* number of length code lengths */
        this.ndist = 0; /* number of distance code lengths */
        this.have = 0; /* number of code lengths in lens[] */
        this.next = null; /* next available space in codes[] */
        this.lens = new utils.Buf16(320); /* temporary storage for code lengths */
        this.work = new utils.Buf16(288); /* work area for code table building */
        /*
         because we don't have pointers in js, we use lencode and distcode directly
         as buffers so we don't need codes
        */
        //this.codes = new utils.Buf32(ENOUGH);       /* space for code tables */
        this.lendyn = null; /* dynamic table for length/literal codes (JS specific) */
        this.distdyn = null; /* dynamic table for distance codes (JS specific) */
        this.sane = 0; /* if false, allow invalid distance too far */
        this.back = 0; /* bits back of last unprocessed length/lit */
        this.was = 0; /* initial length of match */
    }
    function inflateResetKeep(strm) {
        var state;
        if (!strm || !strm.state) {
            return Z_STREAM_ERROR$1;
        }
        state = strm.state;
        strm.total_in = strm.total_out = state.total = 0;
        strm.msg = ''; /*Z_NULL*/
        if (state.wrap) { /* to support ill-conceived Java test suite */
            strm.adler = state.wrap & 1;
        }
        state.mode = HEAD;
        state.last = 0;
        state.havedict = 0;
        state.dmax = 32768;
        state.head = null /*Z_NULL*/;
        state.hold = 0;
        state.bits = 0;
        //state.lencode = state.distcode = state.next = state.codes;
        state.lencode = state.lendyn = new utils.Buf32(ENOUGH_LENS$1);
        state.distcode = state.distdyn = new utils.Buf32(ENOUGH_DISTS$1);
        state.sane = 1;
        state.back = -1;
        //Tracev((stderr, "inflate: reset\n"));
        return Z_OK$2;
    }
    function inflateReset(strm) {
        var state;
        if (!strm || !strm.state) {
            return Z_STREAM_ERROR$1;
        }
        state = strm.state;
        state.wsize = 0;
        state.whave = 0;
        state.wnext = 0;
        return inflateResetKeep(strm);
    }
    function inflateReset2(strm, windowBits) {
        var wrap;
        var state;
        /* get the state */
        if (!strm || !strm.state) {
            return Z_STREAM_ERROR$1;
        }
        state = strm.state;
        /* extract wrap request from windowBits parameter */
        if (windowBits < 0) {
            wrap = 0;
            windowBits = -windowBits;
        }
        else {
            wrap = (windowBits >> 4) + 1;
            if (windowBits < 48) {
                windowBits &= 15;
            }
        }
        /* set number of window bits, free window if different */
        if (windowBits && (windowBits < 8 || windowBits > 15)) {
            return Z_STREAM_ERROR$1;
        }
        if (state.window !== null && state.wbits !== windowBits) {
            state.window = null;
        }
        /* update state and reset the rest of it */
        state.wrap = wrap;
        state.wbits = windowBits;
        return inflateReset(strm);
    }
    function inflateInit2(strm, windowBits) {
        var ret;
        var state;
        if (!strm) {
            return Z_STREAM_ERROR$1;
        }
        //strm.msg = Z_NULL;                 /* in case we return an error */
        state = new InflateState();
        //if (state === Z_NULL) return Z_MEM_ERROR;
        //Tracev((stderr, "inflate: allocated\n"));
        strm.state = state;
        state.window = null /*Z_NULL*/;
        ret = inflateReset2(strm, windowBits);
        if (ret !== Z_OK$2) {
            strm.state = null /*Z_NULL*/;
        }
        return ret;
    }
    function inflateInit(strm) {
        return inflateInit2(strm, DEF_WBITS);
    }
    /*
     Return state with length and distance decoding tables and index sizes set to
     fixed code decoding.  Normally this returns fixed tables from inffixed.h.
     If BUILDFIXED is defined, then instead this routine builds the tables the
     first time it's called, and returns those tables the first time and
     thereafter.  This reduces the size of the code by about 2K bytes, in
     exchange for a little execution time.  However, BUILDFIXED should not be
     used for threaded applications, since the rewriting of the tables and virgin
     may not be thread-safe.
     */
    var virgin = true;
    var lenfix, distfix; // We have no pointers in JS, so keep tables separate
    function fixedtables(state) {
        /* build fixed huffman tables if first call (may not be thread safe) */
        if (virgin) {
            var sym;
            lenfix = new utils.Buf32(512);
            distfix = new utils.Buf32(32);
            /* literal/length table */
            sym = 0;
            while (sym < 144) {
                state.lens[sym++] = 8;
            }
            while (sym < 256) {
                state.lens[sym++] = 9;
            }
            while (sym < 280) {
                state.lens[sym++] = 7;
            }
            while (sym < 288) {
                state.lens[sym++] = 8;
            }
            inflate_table(LENS$1, state.lens, 0, 288, lenfix, 0, state.work, { bits: 9 });
            /* distance table */
            sym = 0;
            while (sym < 32) {
                state.lens[sym++] = 5;
            }
            inflate_table(DISTS$1, state.lens, 0, 32, distfix, 0, state.work, { bits: 5 });
            /* do this just once */
            virgin = false;
        }
        state.lencode = lenfix;
        state.lenbits = 9;
        state.distcode = distfix;
        state.distbits = 5;
    }
    /*
     Update the window with the last wsize (normally 32K) bytes written before
     returning.  If window does not exist yet, create it.  This is only called
     when a window is already in use, or when output has been written during this
     inflate call, but the end of the deflate stream has not been reached yet.
     It is also called to create a window for dictionary data when a dictionary
     is loaded.

     Providing output buffers larger than 32K to inflate() should provide a speed
     advantage, since only the last 32K of output is copied to the sliding window
     upon return from inflate(), and since all distances after the first 32K of
     output will fall in the output data, making match copies simpler and faster.
     The advantage may be dependent on the size of the processor's data caches.
     */
    function updatewindow(strm, src, end, copy) {
        var dist;
        var state = strm.state;
        /* if it hasn't been done already, allocate space for the window */
        if (state.window === null) {
            state.wsize = 1 << state.wbits;
            state.wnext = 0;
            state.whave = 0;
            state.window = new utils.Buf8(state.wsize);
        }
        /* copy state->wsize or less output bytes into the circular window */
        if (copy >= state.wsize) {
            utils.arraySet(state.window, src, end - state.wsize, state.wsize, 0);
            state.wnext = 0;
            state.whave = state.wsize;
        }
        else {
            dist = state.wsize - state.wnext;
            if (dist > copy) {
                dist = copy;
            }
            //zmemcpy(state->window + state->wnext, end - copy, dist);
            utils.arraySet(state.window, src, end - copy, dist, state.wnext);
            copy -= dist;
            if (copy) {
                //zmemcpy(state->window, end - copy, copy);
                utils.arraySet(state.window, src, end - copy, copy, 0);
                state.wnext = copy;
                state.whave = state.wsize;
            }
            else {
                state.wnext += dist;
                if (state.wnext === state.wsize) {
                    state.wnext = 0;
                }
                if (state.whave < state.wsize) {
                    state.whave += dist;
                }
            }
        }
        return 0;
    }
    function inflate(strm, flush) {
        var state;
        var input, output; // input/output buffers
        var next; /* next input INDEX */
        var put; /* next output INDEX */
        var have, left; /* available input and output */
        var hold; /* bit buffer */
        var bits; /* bits in bit buffer */
        var _in, _out; /* save starting available input and output */
        var copy; /* number of stored or match bytes to copy */
        var from; /* where to copy match bytes from */
        var from_source;
        var here = 0; /* current decoding table entry */
        var here_bits, here_op, here_val; // paked "here" denormalized (JS specific)
        //var last;                   /* parent table entry */
        var last_bits, last_op, last_val; // paked "last" denormalized (JS specific)
        var len; /* length to copy for repeats, bits to drop */
        var ret; /* return code */
        var hbuf = new utils.Buf8(4); /* buffer for gzip header crc calculation */
        var opts;
        var n; // temporary var for NEED_BITS
        var order = /* permutation of code lengths */ [16, 17, 18, 0, 8, 7, 9, 6, 10, 5, 11, 4, 12, 3, 13, 2, 14, 1, 15];
        if (!strm || !strm.state || !strm.output ||
            (!strm.input && strm.avail_in !== 0)) {
            return Z_STREAM_ERROR$1;
        }
        state = strm.state;
        if (state.mode === TYPE$1) {
            state.mode = TYPEDO;
        } /* skip check */
        //--- LOAD() ---
        put = strm.next_out;
        output = strm.output;
        left = strm.avail_out;
        next = strm.next_in;
        input = strm.input;
        have = strm.avail_in;
        hold = state.hold;
        bits = state.bits;
        //---
        _in = have;
        _out = left;
        ret = Z_OK$2;
        inf_leave: // goto emulation
         for (;;) {
            switch (state.mode) {
                case HEAD:
                    if (state.wrap === 0) {
                        state.mode = TYPEDO;
                        break;
                    }
                    //=== NEEDBITS(16);
                    while (bits < 16) {
                        if (have === 0) {
                            break inf_leave;
                        }
                        have--;
                        hold += input[next++] << bits;
                        bits += 8;
                    }
                    //===//
                    if ((state.wrap & 2) && hold === 0x8b1f) { /* gzip header */
                        state.check = 0 /*crc32(0L, Z_NULL, 0)*/;
                        //=== CRC2(state.check, hold);
                        hbuf[0] = hold & 0xff;
                        hbuf[1] = (hold >>> 8) & 0xff;
                        state.check = crc32$2(state.check, hbuf, 2, 0);
                        //===//
                        //=== INITBITS();
                        hold = 0;
                        bits = 0;
                        //===//
                        state.mode = FLAGS;
                        break;
                    }
                    state.flags = 0; /* expect zlib header */
                    if (state.head) {
                        state.head.done = false;
                    }
                    if (!(state.wrap & 1) || /* check if zlib header allowed */
                        (((hold & 0xff) /*BITS(8)*/ << 8) + (hold >> 8)) % 31) {
                        strm.msg = 'incorrect header check';
                        state.mode = BAD$1;
                        break;
                    }
                    if ((hold & 0x0f) /*BITS(4)*/ !== Z_DEFLATED$2) {
                        strm.msg = 'unknown compression method';
                        state.mode = BAD$1;
                        break;
                    }
                    //--- DROPBITS(4) ---//
                    hold >>>= 4;
                    bits -= 4;
                    //---//
                    len = (hold & 0x0f) /*BITS(4)*/ + 8;
                    if (state.wbits === 0) {
                        state.wbits = len;
                    }
                    else if (len > state.wbits) {
                        strm.msg = 'invalid window size';
                        state.mode = BAD$1;
                        break;
                    }
                    state.dmax = 1 << len;
                    //Tracev((stderr, "inflate:   zlib header ok\n"));
                    strm.adler = state.check = 1 /*adler32(0L, Z_NULL, 0)*/;
                    state.mode = hold & 0x200 ? DICTID : TYPE$1;
                    //=== INITBITS();
                    hold = 0;
                    bits = 0;
                    //===//
                    break;
                case FLAGS:
                    //=== NEEDBITS(16); */
                    while (bits < 16) {
                        if (have === 0) {
                            break inf_leave;
                        }
                        have--;
                        hold += input[next++] << bits;
                        bits += 8;
                    }
                    //===//
                    state.flags = hold;
                    if ((state.flags & 0xff) !== Z_DEFLATED$2) {
                        strm.msg = 'unknown compression method';
                        state.mode = BAD$1;
                        break;
                    }
                    if (state.flags & 0xe000) {
                        strm.msg = 'unknown header flags set';
                        state.mode = BAD$1;
                        break;
                    }
                    if (state.head) {
                        state.head.text = ((hold >> 8) & 1);
                    }
                    if (state.flags & 0x0200) {
                        //=== CRC2(state.check, hold);
                        hbuf[0] = hold & 0xff;
                        hbuf[1] = (hold >>> 8) & 0xff;
                        state.check = crc32$2(state.check, hbuf, 2, 0);
                        //===//
                    }
                    //=== INITBITS();
                    hold = 0;
                    bits = 0;
                    //===//
                    state.mode = TIME;
                /* falls through */
                case TIME:
                    //=== NEEDBITS(32); */
                    while (bits < 32) {
                        if (have === 0) {
                            break inf_leave;
                        }
                        have--;
                        hold += input[next++] << bits;
                        bits += 8;
                    }
                    //===//
                    if (state.head) {
                        state.head.time = hold;
                    }
                    if (state.flags & 0x0200) {
                        //=== CRC4(state.check, hold)
                        hbuf[0] = hold & 0xff;
                        hbuf[1] = (hold >>> 8) & 0xff;
                        hbuf[2] = (hold >>> 16) & 0xff;
                        hbuf[3] = (hold >>> 24) & 0xff;
                        state.check = crc32$2(state.check, hbuf, 4, 0);
                        //===
                    }
                    //=== INITBITS();
                    hold = 0;
                    bits = 0;
                    //===//
                    state.mode = OS;
                /* falls through */
                case OS:
                    //=== NEEDBITS(16); */
                    while (bits < 16) {
                        if (have === 0) {
                            break inf_leave;
                        }
                        have--;
                        hold += input[next++] << bits;
                        bits += 8;
                    }
                    //===//
                    if (state.head) {
                        state.head.xflags = (hold & 0xff);
                        state.head.os = (hold >> 8);
                    }
                    if (state.flags & 0x0200) {
                        //=== CRC2(state.check, hold);
                        hbuf[0] = hold & 0xff;
                        hbuf[1] = (hold >>> 8) & 0xff;
                        state.check = crc32$2(state.check, hbuf, 2, 0);
                        //===//
                    }
                    //=== INITBITS();
                    hold = 0;
                    bits = 0;
                    //===//
                    state.mode = EXLEN;
                /* falls through */
                case EXLEN:
                    if (state.flags & 0x0400) {
                        //=== NEEDBITS(16); */
                        while (bits < 16) {
                            if (have === 0) {
                                break inf_leave;
                            }
                            have--;
                            hold += input[next++] << bits;
                            bits += 8;
                        }
                        //===//
                        state.length = hold;
                        if (state.head) {
                            state.head.extra_len = hold;
                        }
                        if (state.flags & 0x0200) {
                            //=== CRC2(state.check, hold);
                            hbuf[0] = hold & 0xff;
                            hbuf[1] = (hold >>> 8) & 0xff;
                            state.check = crc32$2(state.check, hbuf, 2, 0);
                            //===//
                        }
                        //=== INITBITS();
                        hold = 0;
                        bits = 0;
                        //===//
                    }
                    else if (state.head) {
                        state.head.extra = null /*Z_NULL*/;
                    }
                    state.mode = EXTRA;
                /* falls through */
                case EXTRA:
                    if (state.flags & 0x0400) {
                        copy = state.length;
                        if (copy > have) {
                            copy = have;
                        }
                        if (copy) {
                            if (state.head) {
                                len = state.head.extra_len - state.length;
                                if (!state.head.extra) {
                                    // Use untyped array for more convenient processing later
                                    state.head.extra = new Array(state.head.extra_len);
                                }
                                utils.arraySet(state.head.extra, input, next, 
                                // extra field is limited to 65536 bytes
                                // - no need for additional size check
                                copy, 
                                /*len + copy > state.head.extra_max - len ? state.head.extra_max : copy,*/
                                len);
                                //zmemcpy(state.head.extra + len, next,
                                //        len + copy > state.head.extra_max ?
                                //        state.head.extra_max - len : copy);
                            }
                            if (state.flags & 0x0200) {
                                state.check = crc32$2(state.check, input, copy, next);
                            }
                            have -= copy;
                            next += copy;
                            state.length -= copy;
                        }
                        if (state.length) {
                            break inf_leave;
                        }
                    }
                    state.length = 0;
                    state.mode = NAME;
                /* falls through */
                case NAME:
                    if (state.flags & 0x0800) {
                        if (have === 0) {
                            break inf_leave;
                        }
                        copy = 0;
                        do {
                            // TODO: 2 or 1 bytes?
                            len = input[next + copy++];
                            /* use constant limit because in js we should not preallocate memory */
                            if (state.head && len &&
                                (state.length < 65536 /*state.head.name_max*/)) {
                                state.head.name += String.fromCharCode(len);
                            }
                        } while (len && copy < have);
                        if (state.flags & 0x0200) {
                            state.check = crc32$2(state.check, input, copy, next);
                        }
                        have -= copy;
                        next += copy;
                        if (len) {
                            break inf_leave;
                        }
                    }
                    else if (state.head) {
                        state.head.name = null;
                    }
                    state.length = 0;
                    state.mode = COMMENT;
                /* falls through */
                case COMMENT:
                    if (state.flags & 0x1000) {
                        if (have === 0) {
                            break inf_leave;
                        }
                        copy = 0;
                        do {
                            len = input[next + copy++];
                            /* use constant limit because in js we should not preallocate memory */
                            if (state.head && len &&
                                (state.length < 65536 /*state.head.comm_max*/)) {
                                state.head.comment += String.fromCharCode(len);
                            }
                        } while (len && copy < have);
                        if (state.flags & 0x0200) {
                            state.check = crc32$2(state.check, input, copy, next);
                        }
                        have -= copy;
                        next += copy;
                        if (len) {
                            break inf_leave;
                        }
                    }
                    else if (state.head) {
                        state.head.comment = null;
                    }
                    state.mode = HCRC;
                /* falls through */
                case HCRC:
                    if (state.flags & 0x0200) {
                        //=== NEEDBITS(16); */
                        while (bits < 16) {
                            if (have === 0) {
                                break inf_leave;
                            }
                            have--;
                            hold += input[next++] << bits;
                            bits += 8;
                        }
                        //===//
                        if (hold !== (state.check & 0xffff)) {
                            strm.msg = 'header crc mismatch';
                            state.mode = BAD$1;
                            break;
                        }
                        //=== INITBITS();
                        hold = 0;
                        bits = 0;
                        //===//
                    }
                    if (state.head) {
                        state.head.hcrc = ((state.flags >> 9) & 1);
                        state.head.done = true;
                    }
                    strm.adler = state.check = 0;
                    state.mode = TYPE$1;
                    break;
                case DICTID:
                    //=== NEEDBITS(32); */
                    while (bits < 32) {
                        if (have === 0) {
                            break inf_leave;
                        }
                        have--;
                        hold += input[next++] << bits;
                        bits += 8;
                    }
                    //===//
                    strm.adler = state.check = zswap32(hold);
                    //=== INITBITS();
                    hold = 0;
                    bits = 0;
                    //===//
                    state.mode = DICT;
                /* falls through */
                case DICT:
                    if (state.havedict === 0) {
                        //--- RESTORE() ---
                        strm.next_out = put;
                        strm.avail_out = left;
                        strm.next_in = next;
                        strm.avail_in = have;
                        state.hold = hold;
                        state.bits = bits;
                        //---
                        return Z_NEED_DICT;
                    }
                    strm.adler = state.check = 1 /*adler32(0L, Z_NULL, 0)*/;
                    state.mode = TYPE$1;
                /* falls through */
                case TYPE$1:
                    if (flush === Z_BLOCK$1 || flush === Z_TREES) {
                        break inf_leave;
                    }
                /* falls through */
                case TYPEDO:
                    if (state.last) {
                        //--- BYTEBITS() ---//
                        hold >>>= bits & 7;
                        bits -= bits & 7;
                        //---//
                        state.mode = CHECK;
                        break;
                    }
                    //=== NEEDBITS(3); */
                    while (bits < 3) {
                        if (have === 0) {
                            break inf_leave;
                        }
                        have--;
                        hold += input[next++] << bits;
                        bits += 8;
                    }
                    //===//
                    state.last = (hold & 0x01) /*BITS(1)*/;
                    //--- DROPBITS(1) ---//
                    hold >>>= 1;
                    bits -= 1;
                    //---//
                    switch ((hold & 0x03) /*BITS(2)*/) {
                        case 0: /* stored block */
                            //Tracev((stderr, "inflate:     stored block%s\n",
                            //        state.last ? " (last)" : ""));
                            state.mode = STORED;
                            break;
                        case 1: /* fixed block */
                            fixedtables(state);
                            //Tracev((stderr, "inflate:     fixed codes block%s\n",
                            //        state.last ? " (last)" : ""));
                            state.mode = LEN_; /* decode codes */
                            if (flush === Z_TREES) {
                                //--- DROPBITS(2) ---//
                                hold >>>= 2;
                                bits -= 2;
                                //---//
                                break inf_leave;
                            }
                            break;
                        case 2: /* dynamic block */
                            //Tracev((stderr, "inflate:     dynamic codes block%s\n",
                            //        state.last ? " (last)" : ""));
                            state.mode = TABLE;
                            break;
                        case 3:
                            strm.msg = 'invalid block type';
                            state.mode = BAD$1;
                    }
                    //--- DROPBITS(2) ---//
                    hold >>>= 2;
                    bits -= 2;
                    //---//
                    break;
                case STORED:
                    //--- BYTEBITS() ---// /* go to byte boundary */
                    hold >>>= bits & 7;
                    bits -= bits & 7;
                    //---//
                    //=== NEEDBITS(32); */
                    while (bits < 32) {
                        if (have === 0) {
                            break inf_leave;
                        }
                        have--;
                        hold += input[next++] << bits;
                        bits += 8;
                    }
                    //===//
                    if ((hold & 0xffff) !== ((hold >>> 16) ^ 0xffff)) {
                        strm.msg = 'invalid stored block lengths';
                        state.mode = BAD$1;
                        break;
                    }
                    state.length = hold & 0xffff;
                    //Tracev((stderr, "inflate:       stored length %u\n",
                    //        state.length));
                    //=== INITBITS();
                    hold = 0;
                    bits = 0;
                    //===//
                    state.mode = COPY_;
                    if (flush === Z_TREES) {
                        break inf_leave;
                    }
                /* falls through */
                case COPY_:
                    state.mode = COPY;
                /* falls through */
                case COPY:
                    copy = state.length;
                    if (copy) {
                        if (copy > have) {
                            copy = have;
                        }
                        if (copy > left) {
                            copy = left;
                        }
                        if (copy === 0) {
                            break inf_leave;
                        }
                        //--- zmemcpy(put, next, copy); ---
                        utils.arraySet(output, input, next, copy, put);
                        //---//
                        have -= copy;
                        next += copy;
                        left -= copy;
                        put += copy;
                        state.length -= copy;
                        break;
                    }
                    //Tracev((stderr, "inflate:       stored end\n"));
                    state.mode = TYPE$1;
                    break;
                case TABLE:
                    //=== NEEDBITS(14); */
                    while (bits < 14) {
                        if (have === 0) {
                            break inf_leave;
                        }
                        have--;
                        hold += input[next++] << bits;
                        bits += 8;
                    }
                    //===//
                    state.nlen = (hold & 0x1f) /*BITS(5)*/ + 257;
                    //--- DROPBITS(5) ---//
                    hold >>>= 5;
                    bits -= 5;
                    //---//
                    state.ndist = (hold & 0x1f) /*BITS(5)*/ + 1;
                    //--- DROPBITS(5) ---//
                    hold >>>= 5;
                    bits -= 5;
                    //---//
                    state.ncode = (hold & 0x0f) /*BITS(4)*/ + 4;
                    //--- DROPBITS(4) ---//
                    hold >>>= 4;
                    bits -= 4;
                    //---//
                    //#ifndef PKZIP_BUG_WORKAROUND
                    if (state.nlen > 286 || state.ndist > 30) {
                        strm.msg = 'too many length or distance symbols';
                        state.mode = BAD$1;
                        break;
                    }
                    //#endif
                    //Tracev((stderr, "inflate:       table sizes ok\n"));
                    state.have = 0;
                    state.mode = LENLENS;
                /* falls through */
                case LENLENS:
                    while (state.have < state.ncode) {
                        //=== NEEDBITS(3);
                        while (bits < 3) {
                            if (have === 0) {
                                break inf_leave;
                            }
                            have--;
                            hold += input[next++] << bits;
                            bits += 8;
                        }
                        //===//
                        state.lens[order[state.have++]] = (hold & 0x07); //BITS(3);
                        //--- DROPBITS(3) ---//
                        hold >>>= 3;
                        bits -= 3;
                        //---//
                    }
                    while (state.have < 19) {
                        state.lens[order[state.have++]] = 0;
                    }
                    // We have separate tables & no pointers. 2 commented lines below not needed.
                    //state.next = state.codes;
                    //state.lencode = state.next;
                    // Switch to use dynamic table
                    state.lencode = state.lendyn;
                    state.lenbits = 7;
                    opts = { bits: state.lenbits };
                    ret = inflate_table(CODES$1, state.lens, 0, 19, state.lencode, 0, state.work, opts);
                    state.lenbits = opts.bits;
                    if (ret) {
                        strm.msg = 'invalid code lengths set';
                        state.mode = BAD$1;
                        break;
                    }
                    //Tracev((stderr, "inflate:       code lengths ok\n"));
                    state.have = 0;
                    state.mode = CODELENS;
                /* falls through */
                case CODELENS:
                    while (state.have < state.nlen + state.ndist) {
                        for (;;) {
                            here = state.lencode[hold & ((1 << state.lenbits) - 1)]; /*BITS(state.lenbits)*/
                            here_bits = here >>> 24;
                            here_op = (here >>> 16) & 0xff;
                            here_val = here & 0xffff;
                            if ((here_bits) <= bits) {
                                break;
                            }
                            //--- PULLBYTE() ---//
                            if (have === 0) {
                                break inf_leave;
                            }
                            have--;
                            hold += input[next++] << bits;
                            bits += 8;
                            //---//
                        }
                        if (here_val < 16) {
                            //--- DROPBITS(here.bits) ---//
                            hold >>>= here_bits;
                            bits -= here_bits;
                            //---//
                            state.lens[state.have++] = here_val;
                        }
                        else {
                            if (here_val === 16) {
                                //=== NEEDBITS(here.bits + 2);
                                n = here_bits + 2;
                                while (bits < n) {
                                    if (have === 0) {
                                        break inf_leave;
                                    }
                                    have--;
                                    hold += input[next++] << bits;
                                    bits += 8;
                                }
                                //===//
                                //--- DROPBITS(here.bits) ---//
                                hold >>>= here_bits;
                                bits -= here_bits;
                                //---//
                                if (state.have === 0) {
                                    strm.msg = 'invalid bit length repeat';
                                    state.mode = BAD$1;
                                    break;
                                }
                                len = state.lens[state.have - 1];
                                copy = 3 + (hold & 0x03); //BITS(2);
                                //--- DROPBITS(2) ---//
                                hold >>>= 2;
                                bits -= 2;
                                //---//
                            }
                            else if (here_val === 17) {
                                //=== NEEDBITS(here.bits + 3);
                                n = here_bits + 3;
                                while (bits < n) {
                                    if (have === 0) {
                                        break inf_leave;
                                    }
                                    have--;
                                    hold += input[next++] << bits;
                                    bits += 8;
                                }
                                //===//
                                //--- DROPBITS(here.bits) ---//
                                hold >>>= here_bits;
                                bits -= here_bits;
                                //---//
                                len = 0;
                                copy = 3 + (hold & 0x07); //BITS(3);
                                //--- DROPBITS(3) ---//
                                hold >>>= 3;
                                bits -= 3;
                                //---//
                            }
                            else {
                                //=== NEEDBITS(here.bits + 7);
                                n = here_bits + 7;
                                while (bits < n) {
                                    if (have === 0) {
                                        break inf_leave;
                                    }
                                    have--;
                                    hold += input[next++] << bits;
                                    bits += 8;
                                }
                                //===//
                                //--- DROPBITS(here.bits) ---//
                                hold >>>= here_bits;
                                bits -= here_bits;
                                //---//
                                len = 0;
                                copy = 11 + (hold & 0x7f); //BITS(7);
                                //--- DROPBITS(7) ---//
                                hold >>>= 7;
                                bits -= 7;
                                //---//
                            }
                            if (state.have + copy > state.nlen + state.ndist) {
                                strm.msg = 'invalid bit length repeat';
                                state.mode = BAD$1;
                                break;
                            }
                            while (copy--) {
                                state.lens[state.have++] = len;
                            }
                        }
                    }
                    /* handle error breaks in while */
                    if (state.mode === BAD$1) {
                        break;
                    }
                    /* check for end-of-block code (better have one) */
                    if (state.lens[256] === 0) {
                        strm.msg = 'invalid code -- missing end-of-block';
                        state.mode = BAD$1;
                        break;
                    }
                    /* build code tables -- note: do not change the lenbits or distbits
                       values here (9 and 6) without reading the comments in inftrees.h
                       concerning the ENOUGH constants, which depend on those values */
                    state.lenbits = 9;
                    opts = { bits: state.lenbits };
                    ret = inflate_table(LENS$1, state.lens, 0, state.nlen, state.lencode, 0, state.work, opts);
                    // We have separate tables & no pointers. 2 commented lines below not needed.
                    // state.next_index = opts.table_index;
                    state.lenbits = opts.bits;
                    // state.lencode = state.next;
                    if (ret) {
                        strm.msg = 'invalid literal/lengths set';
                        state.mode = BAD$1;
                        break;
                    }
                    state.distbits = 6;
                    //state.distcode.copy(state.codes);
                    // Switch to use dynamic table
                    state.distcode = state.distdyn;
                    opts = { bits: state.distbits };
                    ret = inflate_table(DISTS$1, state.lens, state.nlen, state.ndist, state.distcode, 0, state.work, opts);
                    // We have separate tables & no pointers. 2 commented lines below not needed.
                    // state.next_index = opts.table_index;
                    state.distbits = opts.bits;
                    // state.distcode = state.next;
                    if (ret) {
                        strm.msg = 'invalid distances set';
                        state.mode = BAD$1;
                        break;
                    }
                    //Tracev((stderr, 'inflate:       codes ok\n'));
                    state.mode = LEN_;
                    if (flush === Z_TREES) {
                        break inf_leave;
                    }
                /* falls through */
                case LEN_:
                    state.mode = LEN;
                /* falls through */
                case LEN:
                    if (have >= 6 && left >= 258) {
                        //--- RESTORE() ---
                        strm.next_out = put;
                        strm.avail_out = left;
                        strm.next_in = next;
                        strm.avail_in = have;
                        state.hold = hold;
                        state.bits = bits;
                        //---
                        inflate_fast(strm, _out);
                        //--- LOAD() ---
                        put = strm.next_out;
                        output = strm.output;
                        left = strm.avail_out;
                        next = strm.next_in;
                        input = strm.input;
                        have = strm.avail_in;
                        hold = state.hold;
                        bits = state.bits;
                        //---
                        if (state.mode === TYPE$1) {
                            state.back = -1;
                        }
                        break;
                    }
                    state.back = 0;
                    for (;;) {
                        here = state.lencode[hold & ((1 << state.lenbits) - 1)]; /*BITS(state.lenbits)*/
                        here_bits = here >>> 24;
                        here_op = (here >>> 16) & 0xff;
                        here_val = here & 0xffff;
                        if (here_bits <= bits) {
                            break;
                        }
                        //--- PULLBYTE() ---//
                        if (have === 0) {
                            break inf_leave;
                        }
                        have--;
                        hold += input[next++] << bits;
                        bits += 8;
                        //---//
                    }
                    if (here_op && (here_op & 0xf0) === 0) {
                        last_bits = here_bits;
                        last_op = here_op;
                        last_val = here_val;
                        for (;;) {
                            here = state.lencode[last_val +
                                ((hold & ((1 << (last_bits + last_op)) - 1)) /*BITS(last.bits + last.op)*/ >> last_bits)];
                            here_bits = here >>> 24;
                            here_op = (here >>> 16) & 0xff;
                            here_val = here & 0xffff;
                            if ((last_bits + here_bits) <= bits) {
                                break;
                            }
                            //--- PULLBYTE() ---//
                            if (have === 0) {
                                break inf_leave;
                            }
                            have--;
                            hold += input[next++] << bits;
                            bits += 8;
                            //---//
                        }
                        //--- DROPBITS(last.bits) ---//
                        hold >>>= last_bits;
                        bits -= last_bits;
                        //---//
                        state.back += last_bits;
                    }
                    //--- DROPBITS(here.bits) ---//
                    hold >>>= here_bits;
                    bits -= here_bits;
                    //---//
                    state.back += here_bits;
                    state.length = here_val;
                    if (here_op === 0) {
                        //Tracevv((stderr, here.val >= 0x20 && here.val < 0x7f ?
                        //        "inflate:         literal '%c'\n" :
                        //        "inflate:         literal 0x%02x\n", here.val));
                        state.mode = LIT;
                        break;
                    }
                    if (here_op & 32) {
                        //Tracevv((stderr, "inflate:         end of block\n"));
                        state.back = -1;
                        state.mode = TYPE$1;
                        break;
                    }
                    if (here_op & 64) {
                        strm.msg = 'invalid literal/length code';
                        state.mode = BAD$1;
                        break;
                    }
                    state.extra = here_op & 15;
                    state.mode = LENEXT;
                /* falls through */
                case LENEXT:
                    if (state.extra) {
                        //=== NEEDBITS(state.extra);
                        n = state.extra;
                        while (bits < n) {
                            if (have === 0) {
                                break inf_leave;
                            }
                            have--;
                            hold += input[next++] << bits;
                            bits += 8;
                        }
                        //===//
                        state.length += hold & ((1 << state.extra) - 1) /*BITS(state.extra)*/;
                        //--- DROPBITS(state.extra) ---//
                        hold >>>= state.extra;
                        bits -= state.extra;
                        //---//
                        state.back += state.extra;
                    }
                    //Tracevv((stderr, "inflate:         length %u\n", state.length));
                    state.was = state.length;
                    state.mode = DIST;
                /* falls through */
                case DIST:
                    for (;;) {
                        here = state.distcode[hold & ((1 << state.distbits) - 1)]; /*BITS(state.distbits)*/
                        here_bits = here >>> 24;
                        here_op = (here >>> 16) & 0xff;
                        here_val = here & 0xffff;
                        if ((here_bits) <= bits) {
                            break;
                        }
                        //--- PULLBYTE() ---//
                        if (have === 0) {
                            break inf_leave;
                        }
                        have--;
                        hold += input[next++] << bits;
                        bits += 8;
                        //---//
                    }
                    if ((here_op & 0xf0) === 0) {
                        last_bits = here_bits;
                        last_op = here_op;
                        last_val = here_val;
                        for (;;) {
                            here = state.distcode[last_val +
                                ((hold & ((1 << (last_bits + last_op)) - 1)) /*BITS(last.bits + last.op)*/ >> last_bits)];
                            here_bits = here >>> 24;
                            here_op = (here >>> 16) & 0xff;
                            here_val = here & 0xffff;
                            if ((last_bits + here_bits) <= bits) {
                                break;
                            }
                            //--- PULLBYTE() ---//
                            if (have === 0) {
                                break inf_leave;
                            }
                            have--;
                            hold += input[next++] << bits;
                            bits += 8;
                            //---//
                        }
                        //--- DROPBITS(last.bits) ---//
                        hold >>>= last_bits;
                        bits -= last_bits;
                        //---//
                        state.back += last_bits;
                    }
                    //--- DROPBITS(here.bits) ---//
                    hold >>>= here_bits;
                    bits -= here_bits;
                    //---//
                    state.back += here_bits;
                    if (here_op & 64) {
                        strm.msg = 'invalid distance code';
                        state.mode = BAD$1;
                        break;
                    }
                    state.offset = here_val;
                    state.extra = (here_op) & 15;
                    state.mode = DISTEXT;
                /* falls through */
                case DISTEXT:
                    if (state.extra) {
                        //=== NEEDBITS(state.extra);
                        n = state.extra;
                        while (bits < n) {
                            if (have === 0) {
                                break inf_leave;
                            }
                            have--;
                            hold += input[next++] << bits;
                            bits += 8;
                        }
                        //===//
                        state.offset += hold & ((1 << state.extra) - 1) /*BITS(state.extra)*/;
                        //--- DROPBITS(state.extra) ---//
                        hold >>>= state.extra;
                        bits -= state.extra;
                        //---//
                        state.back += state.extra;
                    }
                    //#ifdef INFLATE_STRICT
                    if (state.offset > state.dmax) {
                        strm.msg = 'invalid distance too far back';
                        state.mode = BAD$1;
                        break;
                    }
                    //#endif
                    //Tracevv((stderr, "inflate:         distance %u\n", state.offset));
                    state.mode = MATCH;
                /* falls through */
                case MATCH:
                    if (left === 0) {
                        break inf_leave;
                    }
                    copy = _out - left;
                    if (state.offset > copy) { /* copy from window */
                        copy = state.offset - copy;
                        if (copy > state.whave) {
                            if (state.sane) {
                                strm.msg = 'invalid distance too far back';
                                state.mode = BAD$1;
                                break;
                            }
                            // (!) This block is disabled in zlib defaults,
                            // don't enable it for binary compatibility
                            //#ifdef INFLATE_ALLOW_INVALID_DISTANCE_TOOFAR_ARRR
                            //          Trace((stderr, "inflate.c too far\n"));
                            //          copy -= state.whave;
                            //          if (copy > state.length) { copy = state.length; }
                            //          if (copy > left) { copy = left; }
                            //          left -= copy;
                            //          state.length -= copy;
                            //          do {
                            //            output[put++] = 0;
                            //          } while (--copy);
                            //          if (state.length === 0) { state.mode = LEN; }
                            //          break;
                            //#endif
                        }
                        if (copy > state.wnext) {
                            copy -= state.wnext;
                            from = state.wsize - copy;
                        }
                        else {
                            from = state.wnext - copy;
                        }
                        if (copy > state.length) {
                            copy = state.length;
                        }
                        from_source = state.window;
                    }
                    else { /* copy from output */
                        from_source = output;
                        from = put - state.offset;
                        copy = state.length;
                    }
                    if (copy > left) {
                        copy = left;
                    }
                    left -= copy;
                    state.length -= copy;
                    do {
                        output[put++] = from_source[from++];
                    } while (--copy);
                    if (state.length === 0) {
                        state.mode = LEN;
                    }
                    break;
                case LIT:
                    if (left === 0) {
                        break inf_leave;
                    }
                    output[put++] = state.length;
                    left--;
                    state.mode = LEN;
                    break;
                case CHECK:
                    if (state.wrap) {
                        //=== NEEDBITS(32);
                        while (bits < 32) {
                            if (have === 0) {
                                break inf_leave;
                            }
                            have--;
                            // Use '|' instead of '+' to make sure that result is signed
                            hold |= input[next++] << bits;
                            bits += 8;
                        }
                        //===//
                        _out -= left;
                        strm.total_out += _out;
                        state.total += _out;
                        if (_out) {
                            strm.adler = state.check =
                                /*UPDATE(state.check, put - _out, _out);*/
                                (state.flags ? crc32$2(state.check, output, _out, put - _out) : adler32$2(state.check, output, _out, put - _out));
                        }
                        _out = left;
                        // NB: crc32 stored as signed 32-bit int, zswap32 returns signed too
                        if ((state.flags ? hold : zswap32(hold)) !== state.check) {
                            strm.msg = 'incorrect data check';
                            state.mode = BAD$1;
                            break;
                        }
                        //=== INITBITS();
                        hold = 0;
                        bits = 0;
                        //===//
                        //Tracev((stderr, "inflate:   check matches trailer\n"));
                    }
                    state.mode = LENGTH;
                /* falls through */
                case LENGTH:
                    if (state.wrap && state.flags) {
                        //=== NEEDBITS(32);
                        while (bits < 32) {
                            if (have === 0) {
                                break inf_leave;
                            }
                            have--;
                            hold += input[next++] << bits;
                            bits += 8;
                        }
                        //===//
                        if (hold !== (state.total & 0xffffffff)) {
                            strm.msg = 'incorrect length check';
                            state.mode = BAD$1;
                            break;
                        }
                        //=== INITBITS();
                        hold = 0;
                        bits = 0;
                        //===//
                        //Tracev((stderr, "inflate:   length matches trailer\n"));
                    }
                    state.mode = DONE;
                /* falls through */
                case DONE:
                    ret = Z_STREAM_END$2;
                    break inf_leave;
                case BAD$1:
                    ret = Z_DATA_ERROR$1;
                    break inf_leave;
                case MEM:
                    return Z_MEM_ERROR;
                case SYNC:
                /* falls through */
                default:
                    return Z_STREAM_ERROR$1;
            }
        }
        // inf_leave <- here is real place for "goto inf_leave", emulated via "break inf_leave"
        /*
           Return from inflate(), updating the total counts and the check value.
           If there was no progress during the inflate() call, return a buffer
           error.  Call updatewindow() to create and/or update the window state.
           Note: a memory error from inflate() is non-recoverable.
         */
        //--- RESTORE() ---
        strm.next_out = put;
        strm.avail_out = left;
        strm.next_in = next;
        strm.avail_in = have;
        state.hold = hold;
        state.bits = bits;
        //---
        if (state.wsize || (_out !== strm.avail_out && state.mode < BAD$1 &&
            (state.mode < CHECK || flush !== Z_FINISH$2))) {
            if (updatewindow(strm, strm.output, strm.next_out, _out - strm.avail_out)) ;
        }
        _in -= strm.avail_in;
        _out -= strm.avail_out;
        strm.total_in += _in;
        strm.total_out += _out;
        state.total += _out;
        if (state.wrap && _out) {
            strm.adler = state.check = /*UPDATE(state.check, strm.next_out - _out, _out);*/
                (state.flags ? crc32$2(state.check, output, _out, strm.next_out - _out) : adler32$2(state.check, output, _out, strm.next_out - _out));
        }
        strm.data_type = state.bits + (state.last ? 64 : 0) +
            (state.mode === TYPE$1 ? 128 : 0) +
            (state.mode === LEN_ || state.mode === COPY_ ? 256 : 0);
        if (((_in === 0 && _out === 0) || flush === Z_FINISH$2) && ret === Z_OK$2) {
            ret = Z_BUF_ERROR$1;
        }
        return ret;
    }
    function inflateEnd(strm) {
        if (!strm || !strm.state /*|| strm->zfree == (free_func)0*/) {
            return Z_STREAM_ERROR$1;
        }
        var state = strm.state;
        if (state.window) {
            state.window = null;
        }
        strm.state = null;
        return Z_OK$2;
    }
    function inflateGetHeader(strm, head) {
        var state;
        /* check state */
        if (!strm || !strm.state) {
            return Z_STREAM_ERROR$1;
        }
        state = strm.state;
        if ((state.wrap & 2) === 0) {
            return Z_STREAM_ERROR$1;
        }
        /* save header structure */
        state.head = head;
        head.done = false;
        return Z_OK$2;
    }
    function inflateSetDictionary(strm, dictionary) {
        var dictLength = dictionary.length;
        var state;
        var dictid;
        var ret;
        /* check state */
        if (!strm /* == Z_NULL */ || !strm.state /* == Z_NULL */) {
            return Z_STREAM_ERROR$1;
        }
        state = strm.state;
        if (state.wrap !== 0 && state.mode !== DICT) {
            return Z_STREAM_ERROR$1;
        }
        /* check for correct dictionary identifier */
        if (state.mode === DICT) {
            dictid = 1; /* adler32(0, null, 0)*/
            /* dictid = adler32(dictid, dictionary, dictLength); */
            dictid = adler32$2(dictid, dictionary, dictLength, 0);
            if (dictid !== state.check) {
                return Z_DATA_ERROR$1;
            }
        }
        /* copy dictionary to window using updatewindow(), which will amend the
         existing dictionary if appropriate */
        ret = updatewindow(strm, dictionary, dictLength, dictLength);
        if (ret) {
            state.mode = MEM;
            return Z_MEM_ERROR;
        }
        state.havedict = 1;
        // Tracev((stderr, "inflate:   dictionary set\n"));
        return Z_OK$2;
    }
    var inflateReset_1 = inflateReset;
    var inflateReset2_1 = inflateReset2;
    var inflateResetKeep_1 = inflateResetKeep;
    var inflateInit_1 = inflateInit;
    var inflateInit2_1 = inflateInit2;
    var inflate_2 = inflate;
    var inflateEnd_1 = inflateEnd;
    var inflateGetHeader_1 = inflateGetHeader;
    var inflateSetDictionary_1 = inflateSetDictionary;
    var inflateInfo = 'pako inflate (from Nodeca project)';
    /* Not implemented
    exports.inflateCopy = inflateCopy;
    exports.inflateGetDictionary = inflateGetDictionary;
    exports.inflateMark = inflateMark;
    exports.inflatePrime = inflatePrime;
    exports.inflateSync = inflateSync;
    exports.inflateSyncPoint = inflateSyncPoint;
    exports.inflateUndermine = inflateUndermine;
    */
    var inflate_1 = {
        inflateReset: inflateReset_1,
        inflateReset2: inflateReset2_1,
        inflateResetKeep: inflateResetKeep_1,
        inflateInit: inflateInit_1,
        inflateInit2: inflateInit2_1,
        inflate: inflate_2,
        inflateEnd: inflateEnd_1,
        inflateGetHeader: inflateGetHeader_1,
        inflateSetDictionary: inflateSetDictionary_1,
        inflateInfo: inflateInfo
    };

    var inflate$1 = /*#__PURE__*/Object.freeze({
        default: inflate_1,
        __moduleExports: inflate_1,
        inflateReset: inflateReset_1,
        inflateReset2: inflateReset2_1,
        inflateResetKeep: inflateResetKeep_1,
        inflateInit: inflateInit_1,
        inflateInit2: inflateInit2_1,
        inflate: inflate_2,
        inflateEnd: inflateEnd_1,
        inflateGetHeader: inflateGetHeader_1,
        inflateSetDictionary: inflateSetDictionary_1,
        inflateInfo: inflateInfo
    });

    // (C) 1995-2013 Jean-loup Gailly and Mark Adler
    // (C) 2014-2017 Vitaly Puzrin and Andrey Tupitsin
    //
    // This software is provided 'as-is', without any express or implied
    // warranty. In no event will the authors be held liable for any damages
    // arising from the use of this software.
    //
    // Permission is granted to anyone to use this software for any purpose,
    // including commercial applications, and to alter it and redistribute it
    // freely, subject to the following restrictions:
    //
    // 1. The origin of this software must not be misrepresented; you must not
    //   claim that you wrote the original software. If you use this software
    //   in a product, an acknowledgment in the product documentation would be
    //   appreciated but is not required.
    // 2. Altered source versions must be plainly marked as such, and must not be
    //   misrepresented as being the original software.
    // 3. This notice may not be removed or altered from any source distribution.
    var constants = {
        /* Allowed flush values; see deflate() and inflate() below for details */
        Z_NO_FLUSH: 0,
        Z_PARTIAL_FLUSH: 1,
        Z_SYNC_FLUSH: 2,
        Z_FULL_FLUSH: 3,
        Z_FINISH: 4,
        Z_BLOCK: 5,
        Z_TREES: 6,
        /* Return codes for the compression/decompression functions. Negative values
        * are errors, positive values are used for special but normal events.
        */
        Z_OK: 0,
        Z_STREAM_END: 1,
        Z_NEED_DICT: 2,
        Z_ERRNO: -1,
        Z_STREAM_ERROR: -2,
        Z_DATA_ERROR: -3,
        //Z_MEM_ERROR:     -4,
        Z_BUF_ERROR: -5,
        //Z_VERSION_ERROR: -6,
        /* compression levels */
        Z_NO_COMPRESSION: 0,
        Z_BEST_SPEED: 1,
        Z_BEST_COMPRESSION: 9,
        Z_DEFAULT_COMPRESSION: -1,
        Z_FILTERED: 1,
        Z_HUFFMAN_ONLY: 2,
        Z_RLE: 3,
        Z_FIXED: 4,
        Z_DEFAULT_STRATEGY: 0,
        /* Possible values of the data_type field (though see inflate()) */
        Z_BINARY: 0,
        Z_TEXT: 1,
        //Z_ASCII:                1, // = Z_TEXT (deprecated)
        Z_UNKNOWN: 2,
        /* The deflate compression method */
        Z_DEFLATED: 8
        //Z_NULL:                 null // Use -1 or null inline, depending on var type
    };
    var constants_1 = constants.Z_NO_FLUSH;
    var constants_2 = constants.Z_PARTIAL_FLUSH;
    var constants_3 = constants.Z_SYNC_FLUSH;
    var constants_4 = constants.Z_FULL_FLUSH;
    var constants_5 = constants.Z_FINISH;
    var constants_6 = constants.Z_BLOCK;
    var constants_7 = constants.Z_TREES;
    var constants_8 = constants.Z_OK;
    var constants_9 = constants.Z_STREAM_END;
    var constants_10 = constants.Z_NEED_DICT;
    var constants_11 = constants.Z_ERRNO;
    var constants_12 = constants.Z_STREAM_ERROR;
    var constants_13 = constants.Z_DATA_ERROR;
    var constants_14 = constants.Z_BUF_ERROR;
    var constants_15 = constants.Z_NO_COMPRESSION;
    var constants_16 = constants.Z_BEST_SPEED;
    var constants_17 = constants.Z_BEST_COMPRESSION;
    var constants_18 = constants.Z_DEFAULT_COMPRESSION;
    var constants_19 = constants.Z_FILTERED;
    var constants_20 = constants.Z_HUFFMAN_ONLY;
    var constants_21 = constants.Z_RLE;
    var constants_22 = constants.Z_FIXED;
    var constants_23 = constants.Z_DEFAULT_STRATEGY;
    var constants_24 = constants.Z_BINARY;
    var constants_25 = constants.Z_TEXT;
    var constants_26 = constants.Z_UNKNOWN;
    var constants_27 = constants.Z_DEFLATED;

    var constants$1 = /*#__PURE__*/Object.freeze({
        default: constants,
        __moduleExports: constants,
        Z_NO_FLUSH: constants_1,
        Z_PARTIAL_FLUSH: constants_2,
        Z_SYNC_FLUSH: constants_3,
        Z_FULL_FLUSH: constants_4,
        Z_FINISH: constants_5,
        Z_BLOCK: constants_6,
        Z_TREES: constants_7,
        Z_OK: constants_8,
        Z_STREAM_END: constants_9,
        Z_NEED_DICT: constants_10,
        Z_ERRNO: constants_11,
        Z_STREAM_ERROR: constants_12,
        Z_DATA_ERROR: constants_13,
        Z_BUF_ERROR: constants_14,
        Z_NO_COMPRESSION: constants_15,
        Z_BEST_SPEED: constants_16,
        Z_BEST_COMPRESSION: constants_17,
        Z_DEFAULT_COMPRESSION: constants_18,
        Z_FILTERED: constants_19,
        Z_HUFFMAN_ONLY: constants_20,
        Z_RLE: constants_21,
        Z_FIXED: constants_22,
        Z_DEFAULT_STRATEGY: constants_23,
        Z_BINARY: constants_24,
        Z_TEXT: constants_25,
        Z_UNKNOWN: constants_26,
        Z_DEFLATED: constants_27
    });

    // (C) 1995-2013 Jean-loup Gailly and Mark Adler
    // (C) 2014-2017 Vitaly Puzrin and Andrey Tupitsin
    //
    // This software is provided 'as-is', without any express or implied
    // warranty. In no event will the authors be held liable for any damages
    // arising from the use of this software.
    //
    // Permission is granted to anyone to use this software for any purpose,
    // including commercial applications, and to alter it and redistribute it
    // freely, subject to the following restrictions:
    //
    // 1. The origin of this software must not be misrepresented; you must not
    //   claim that you wrote the original software. If you use this software
    //   in a product, an acknowledgment in the product documentation would be
    //   appreciated but is not required.
    // 2. Altered source versions must be plainly marked as such, and must not be
    //   misrepresented as being the original software.
    // 3. This notice may not be removed or altered from any source distribution.
    function GZheader() {
        /* true if compressed data believed to be text */
        this.text = 0;
        /* modification time */
        this.time = 0;
        /* extra flags (not used when writing a gzip file) */
        this.xflags = 0;
        /* operating system */
        this.os = 0;
        /* pointer to extra field or Z_NULL if none */
        this.extra = null;
        /* extra field length (valid if extra != Z_NULL) */
        this.extra_len = 0; // Actually, we don't need it in JS,
        // but leave for few code modifications
        //
        // Setup limits is not necessary because in js we should not preallocate memory
        // for inflate use constant limit in 65536 bytes
        //
        /* space at extra (only when reading header) */
        // this.extra_max  = 0;
        /* pointer to zero-terminated file name or Z_NULL */
        this.name = '';
        /* space at name (only when reading header) */
        // this.name_max   = 0;
        /* pointer to zero-terminated comment or Z_NULL */
        this.comment = '';
        /* space at comment (only when reading header) */
        // this.comm_max   = 0;
        /* true if there was or will be a header crc */
        this.hcrc = 0;
        /* true when done reading gzip header (not used when writing a gzip file) */
        this.done = false;
    }
    var gzheader = GZheader;

    var gzheader$1 = /*#__PURE__*/Object.freeze({
        default: gzheader,
        __moduleExports: gzheader
    });

    var zlib_inflate = (inflate$1 && inflate_1) || inflate$1;

    var c = (constants$1 && constants) || constants$1;

    var GZheader$1 = (gzheader$1 && gzheader) || gzheader$1;

    var toString$1 = Object.prototype.toString;
    /**
     * class Inflate
     *
     * Generic JS-style wrapper for zlib calls. If you don't need
     * streaming behaviour - use more simple functions: [[inflate]]
     * and [[inflateRaw]].
     **/
    /* internal
     * inflate.chunks -> Array
     *
     * Chunks of output data, if [[Inflate#onData]] not overridden.
     **/
    /**
     * Inflate.result -> Uint8Array|Array|String
     *
     * Uncompressed result, generated by default [[Inflate#onData]]
     * and [[Inflate#onEnd]] handlers. Filled after you push last chunk
     * (call [[Inflate#push]] with `Z_FINISH` / `true` param) or if you
     * push a chunk with explicit flush (call [[Inflate#push]] with
     * `Z_SYNC_FLUSH` param).
     **/
    /**
     * Inflate.err -> Number
     *
     * Error code after inflate finished. 0 (Z_OK) on success.
     * Should be checked if broken data possible.
     **/
    /**
     * Inflate.msg -> String
     *
     * Error message, if [[Inflate.err]] != 0
     **/
    /**
     * new Inflate(options)
     * - options (Object): zlib inflate options.
     *
     * Creates new inflator instance with specified params. Throws exception
     * on bad params. Supported options:
     *
     * - `windowBits`
     * - `dictionary`
     *
     * [http://zlib.net/manual.html#Advanced](http://zlib.net/manual.html#Advanced)
     * for more information on these.
     *
     * Additional options, for internal needs:
     *
     * - `chunkSize` - size of generated data chunks (16K by default)
     * - `raw` (Boolean) - do raw inflate
     * - `to` (String) - if equal to 'string', then result will be converted
     *   from utf8 to utf16 (javascript) string. When string output requested,
     *   chunk length can differ from `chunkSize`, depending on content.
     *
     * By default, when no options set, autodetect deflate/gzip data format via
     * wrapper header.
     *
     * ##### Example:
     *
     * ```javascript
     * var pako = require('pako')
     *   , chunk1 = Uint8Array([1,2,3,4,5,6,7,8,9])
     *   , chunk2 = Uint8Array([10,11,12,13,14,15,16,17,18,19]);
     *
     * var inflate = new pako.Inflate({ level: 3});
     *
     * inflate.push(chunk1, false);
     * inflate.push(chunk2, true);  // true -> last chunk
     *
     * if (inflate.err) { throw new Error(inflate.err); }
     *
     * console.log(inflate.result);
     * ```
     **/
    function Inflate(options) {
        if (!(this instanceof Inflate))
            return new Inflate(options);
        this.options = utils.assign({
            chunkSize: 16384,
            windowBits: 0,
            to: ''
        }, options || {});
        var opt = this.options;
        // Force window size for `raw` data, if not set directly,
        // because we have no header for autodetect.
        if (opt.raw && (opt.windowBits >= 0) && (opt.windowBits < 16)) {
            opt.windowBits = -opt.windowBits;
            if (opt.windowBits === 0) {
                opt.windowBits = -15;
            }
        }
        // If `windowBits` not defined (and mode not raw) - set autodetect flag for gzip/deflate
        if ((opt.windowBits >= 0) && (opt.windowBits < 16) &&
            !(options && options.windowBits)) {
            opt.windowBits += 32;
        }
        // Gzip header has no info about windows size, we can do autodetect only
        // for deflate. So, if window size not set, force it to max when gzip possible
        if ((opt.windowBits > 15) && (opt.windowBits < 48)) {
            // bit 3 (16) -> gzipped data
            // bit 4 (32) -> autodetect gzip/deflate
            if ((opt.windowBits & 15) === 0) {
                opt.windowBits |= 15;
            }
        }
        this.err = 0; // error code, if happens (0 = Z_OK)
        this.msg = ''; // error message
        this.ended = false; // used to avoid multiple onEnd() calls
        this.chunks = []; // chunks of compressed data
        this.strm = new ZStream$1();
        this.strm.avail_out = 0;
        var status = zlib_inflate.inflateInit2(this.strm, opt.windowBits);
        if (status !== c.Z_OK) {
            throw new Error(msg[status]);
        }
        this.header = new GZheader$1();
        zlib_inflate.inflateGetHeader(this.strm, this.header);
        // Setup dictionary
        if (opt.dictionary) {
            // Convert data if needed
            if (typeof opt.dictionary === 'string') {
                opt.dictionary = strings$2.string2buf(opt.dictionary);
            }
            else if (toString$1.call(opt.dictionary) === '[object ArrayBuffer]') {
                opt.dictionary = new Uint8Array(opt.dictionary);
            }
            if (opt.raw) { //In raw mode we need to set the dictionary early
                status = zlib_inflate.inflateSetDictionary(this.strm, opt.dictionary);
                if (status !== c.Z_OK) {
                    throw new Error(msg[status]);
                }
            }
        }
    }
    /**
     * Inflate#push(data[, mode]) -> Boolean
     * - data (Uint8Array|Array|ArrayBuffer|String): input data
     * - mode (Number|Boolean): 0..6 for corresponding Z_NO_FLUSH..Z_TREE modes.
     *   See constants. Skipped or `false` means Z_NO_FLUSH, `true` means Z_FINISH.
     *
     * Sends input data to inflate pipe, generating [[Inflate#onData]] calls with
     * new output chunks. Returns `true` on success. The last data block must have
     * mode Z_FINISH (or `true`). That will flush internal pending buffers and call
     * [[Inflate#onEnd]]. For interim explicit flushes (without ending the stream) you
     * can use mode Z_SYNC_FLUSH, keeping the decompression context.
     *
     * On fail call [[Inflate#onEnd]] with error code and return false.
     *
     * We strongly recommend to use `Uint8Array` on input for best speed (output
     * format is detected automatically). Also, don't skip last param and always
     * use the same type in your code (boolean or number). That will improve JS speed.
     *
     * For regular `Array`-s make sure all elements are [0..255].
     *
     * ##### Example
     *
     * ```javascript
     * push(chunk, false); // push one of data chunks
     * ...
     * push(chunk, true);  // push last chunk
     * ```
     **/
    Inflate.prototype.push = function (data, mode) {
        var strm = this.strm;
        var chunkSize = this.options.chunkSize;
        var dictionary = this.options.dictionary;
        var status, _mode;
        var next_out_utf8, tail, utf8str;
        // Flag to properly process Z_BUF_ERROR on testing inflate call
        // when we check that all output data was flushed.
        var allowBufError = false;
        if (this.ended) {
            return false;
        }
        _mode = (mode === ~~mode) ? mode : ((mode === true) ? c.Z_FINISH : c.Z_NO_FLUSH);
        // Convert data if needed
        if (typeof data === 'string') {
            // Only binary strings can be decompressed on practice
            strm.input = strings$2.binstring2buf(data);
        }
        else if (toString$1.call(data) === '[object ArrayBuffer]') {
            strm.input = new Uint8Array(data);
        }
        else {
            strm.input = data;
        }
        strm.next_in = 0;
        strm.avail_in = strm.input.length;
        do {
            if (strm.avail_out === 0) {
                strm.output = new utils.Buf8(chunkSize);
                strm.next_out = 0;
                strm.avail_out = chunkSize;
            }
            status = zlib_inflate.inflate(strm, c.Z_NO_FLUSH); /* no bad return value */
            if (status === c.Z_NEED_DICT && dictionary) {
                status = zlib_inflate.inflateSetDictionary(this.strm, dictionary);
            }
            if (status === c.Z_BUF_ERROR && allowBufError === true) {
                status = c.Z_OK;
                allowBufError = false;
            }
            if (status !== c.Z_STREAM_END && status !== c.Z_OK) {
                this.onEnd(status);
                this.ended = true;
                return false;
            }
            if (strm.next_out) {
                if (strm.avail_out === 0 || status === c.Z_STREAM_END || (strm.avail_in === 0 && (_mode === c.Z_FINISH || _mode === c.Z_SYNC_FLUSH))) {
                    if (this.options.to === 'string') {
                        next_out_utf8 = strings$2.utf8border(strm.output, strm.next_out);
                        tail = strm.next_out - next_out_utf8;
                        utf8str = strings$2.buf2string(strm.output, next_out_utf8);
                        // move tail
                        strm.next_out = tail;
                        strm.avail_out = chunkSize - tail;
                        if (tail) {
                            utils.arraySet(strm.output, strm.output, next_out_utf8, tail, 0);
                        }
                        this.onData(utf8str);
                    }
                    else {
                        this.onData(utils.shrinkBuf(strm.output, strm.next_out));
                    }
                }
            }
            // When no more input data, we should check that internal inflate buffers
            // are flushed. The only way to do it when avail_out = 0 - run one more
            // inflate pass. But if output data not exists, inflate return Z_BUF_ERROR.
            // Here we set flag to process this error properly.
            //
            // NOTE. Deflate does not return error in this case and does not needs such
            // logic.
            if (strm.avail_in === 0 && strm.avail_out === 0) {
                allowBufError = true;
            }
        } while ((strm.avail_in > 0 || strm.avail_out === 0) && status !== c.Z_STREAM_END);
        if (status === c.Z_STREAM_END) {
            _mode = c.Z_FINISH;
        }
        // Finalize on the last chunk.
        if (_mode === c.Z_FINISH) {
            status = zlib_inflate.inflateEnd(this.strm);
            this.onEnd(status);
            this.ended = true;
            return status === c.Z_OK;
        }
        // callback interim results if Z_SYNC_FLUSH.
        if (_mode === c.Z_SYNC_FLUSH) {
            this.onEnd(c.Z_OK);
            strm.avail_out = 0;
            return true;
        }
        return true;
    };
    /**
     * Inflate#onData(chunk) -> Void
     * - chunk (Uint8Array|Array|String): output data. Type of array depends
     *   on js engine support. When string output requested, each chunk
     *   will be string.
     *
     * By default, stores data blocks in `chunks[]` property and glue
     * those in `onEnd`. Override this handler, if you need another behaviour.
     **/
    Inflate.prototype.onData = function (chunk) {
        this.chunks.push(chunk);
    };
    /**
     * Inflate#onEnd(status) -> Void
     * - status (Number): inflate status. 0 (Z_OK) on success,
     *   other if not.
     *
     * Called either after you tell inflate that the input stream is
     * complete (Z_FINISH) or should be flushed (Z_SYNC_FLUSH)
     * or if an error happened. By default - join collected chunks,
     * free memory and fill `results` / `err` properties.
     **/
    Inflate.prototype.onEnd = function (status) {
        // On success - join
        if (status === c.Z_OK) {
            if (this.options.to === 'string') {
                // Glue & convert here, until we teach pako to send
                // utf8 aligned strings to onData
                this.result = this.chunks.join('');
            }
            else {
                this.result = utils.flattenChunks(this.chunks);
            }
        }
        this.chunks = [];
        this.err = status;
        this.msg = this.strm.msg;
    };
    /**
     * inflate(data[, options]) -> Uint8Array|Array|String
     * - data (Uint8Array|Array|String): input data to decompress.
     * - options (Object): zlib inflate options.
     *
     * Decompress `data` with inflate/ungzip and `options`. Autodetect
     * format via wrapper header by default. That's why we don't provide
     * separate `ungzip` method.
     *
     * Supported options are:
     *
     * - windowBits
     *
     * [http://zlib.net/manual.html#Advanced](http://zlib.net/manual.html#Advanced)
     * for more information.
     *
     * Sugar (options):
     *
     * - `raw` (Boolean) - say that we work with raw stream, if you don't wish to specify
     *   negative windowBits implicitly.
     * - `to` (String) - if equal to 'string', then result will be converted
     *   from utf8 to utf16 (javascript) string. When string output requested,
     *   chunk length can differ from `chunkSize`, depending on content.
     *
     *
     * ##### Example:
     *
     * ```javascript
     * var pako = require('pako')
     *   , input = pako.deflate([1,2,3,4,5,6,7,8,9])
     *   , output;
     *
     * try {
     *   output = pako.inflate(input);
     * } catch (err)
     *   console.log(err);
     * }
     * ```
     **/
    function inflate$2(input, options) {
        var inflator = new Inflate(options);
        inflator.push(input, true);
        // That will never happens, if you don't cheat with options :)
        if (inflator.err) {
            throw inflator.msg || msg[inflator.err];
        }
        return inflator.result;
    }
    /**
     * inflateRaw(data[, options]) -> Uint8Array|Array|String
     * - data (Uint8Array|Array|String): input data to decompress.
     * - options (Object): zlib inflate options.
     *
     * The same as [[inflate]], but creates raw data, without wrapper
     * (header and adler32 crc).
     **/
    function inflateRaw(input, options) {
        options = options || {};
        options.raw = true;
        return inflate$2(input, options);
    }
    /**
     * ungzip(data[, options]) -> Uint8Array|Array|String
     * - data (Uint8Array|Array|String): input data to decompress.
     * - options (Object): zlib inflate options.
     *
     * Just shortcut to [[inflate]], because it autodetects format
     * by header.content. Done for convenience.
     **/
    var Inflate_1 = Inflate;
    var inflate_2$1 = inflate$2;
    var inflateRaw_1 = inflateRaw;
    var ungzip = inflate$2;
    var inflate_1$1 = {
        Inflate: Inflate_1,
        inflate: inflate_2$1,
        inflateRaw: inflateRaw_1,
        ungzip: ungzip
    };

    var inflate$3 = /*#__PURE__*/Object.freeze({
        default: inflate_1$1,
        __moduleExports: inflate_1$1,
        Inflate: Inflate_1,
        inflate: inflate_2$1,
        inflateRaw: inflateRaw_1,
        ungzip: ungzip
    });

    var deflate$4 = (deflate$3 && deflate_1$1) || deflate$3;

    var inflate$4 = (inflate$3 && inflate_1$1) || inflate$3;

    var assign = utils.assign;
    var pako = {};
    assign(pako, deflate$4, inflate$4, c);
    var pako_1 = pako;

    var UPNG = {};
    if (Uint8Array && !Uint8Array.prototype.slice) {
        Uint8Array.prototype.slice = function () {
            var arg = [];
            for (var _i = 0; _i < arguments.length; _i++) {
                arg[_i] = arguments[_i];
            }
            var _a;
            return (_a = new Uint8Array(this)).subarray.apply(_a, arg);
        };
    }
    (function (UPNG, pako) {
        UPNG.toRGBA8 = function (out) {
            var w = out.width, h = out.height;
            if (out.tabs.acTL == null)
                return [UPNG.toRGBA8.decodeImage(out.data, w, h, out).buffer];
            var frms = [];
            if (out.frames[0].data == null)
                out.frames[0].data = out.data;
            var img, empty = new Uint8Array(w * h * 4);
            for (var i = 0; i < out.frames.length; i++) {
                var frm = out.frames[i];
                var fx = frm.rect.x, fy = frm.rect.y, fw = frm.rect.width, fh = frm.rect.height;
                var fdata = UPNG.toRGBA8.decodeImage(frm.data, fw, fh, out);
                if (i == 0)
                    img = fdata;
                else if (frm.blend == 0)
                    UPNG._copyTile(fdata, fw, fh, img, w, h, fx, fy, 0);
                else if (frm.blend == 1)
                    UPNG._copyTile(fdata, fw, fh, img, w, h, fx, fy, 1);
                frms.push(img.buffer);
                img = img.slice(0);
                if (frm.dispose == 0) ;
                else if (frm.dispose == 1)
                    UPNG._copyTile(empty, fw, fh, img, w, h, fx, fy, 0);
                else if (frm.dispose == 2) {
                    var pi = i - 1;
                    while (out.frames[pi].dispose == 2)
                        pi--;
                    img = new Uint8Array(frms[pi]).slice(0);
                }
            }
            return frms;
        };
        UPNG.toRGBA8.decodeImage = function (data, w, h, out) {
            var area = w * h, bpp = UPNG.decode._getBPP(out);
            var bpl = Math.ceil(w * bpp / 8); // bytes per line
            var bf = new Uint8Array(area * 4), bf32 = new Uint32Array(bf.buffer);
            var ctype = out.ctype, depth = out.depth;
            var rs = UPNG._bin.readUshort;
            //console.log(ctype, depth);
            if (ctype == 6) { // RGB + alpha
                var qarea = area << 2;
                if (depth == 8)
                    for (var i = 0; i < qarea; i++) {
                        bf[i] = data[i];
                        /*if((i&3)==3 && data[i]!=0) bf[i]=255;*/
                    }
                if (depth == 16)
                    for (var i = 0; i < qarea; i++) {
                        bf[i] = data[i << 1];
                    }
            }
            else if (ctype == 2) { // RGB
                var ts = out.tabs["tRNS"], tr = -1, tg = -1, tb = -1;
                if (ts) {
                    tr = ts[0];
                    tg = ts[1];
                    tb = ts[2];
                }
                if (depth == 8)
                    for (var i = 0; i < area; i++) {
                        var qi = i << 2, ti = i * 3;
                        bf[qi] = data[ti];
                        bf[qi + 1] = data[ti + 1];
                        bf[qi + 2] = data[ti + 2];
                        bf[qi + 3] = 255;
                        if (tr != -1 && data[ti] == tr && data[ti + 1] == tg && data[ti + 2] == tb)
                            bf[qi + 3] = 0;
                    }
                if (depth == 16)
                    for (var i = 0; i < area; i++) {
                        var qi = i << 2, ti = i * 6;
                        bf[qi] = data[ti];
                        bf[qi + 1] = data[ti + 2];
                        bf[qi + 2] = data[ti + 4];
                        bf[qi + 3] = 255;
                        if (tr != -1 && rs(data, ti) == tr && rs(data, ti + 2) == tg && rs(data, ti + 4) == tb)
                            bf[qi + 3] = 0;
                    }
            }
            else if (ctype == 3) { // palette
                var p = out.tabs["PLTE"], ap = out.tabs["tRNS"], tl = ap ? ap.length : 0;
                //console.log(p, ap);
                if (depth == 1)
                    for (var y = 0; y < h; y++) {
                        var s0 = y * bpl, t0 = y * w;
                        for (var i = 0; i < w; i++) {
                            var qi = (t0 + i) << 2, j = ((data[s0 + (i >> 3)] >> (7 - ((i & 7) << 0))) & 1), cj = 3 * j;
                            bf[qi] = p[cj];
                            bf[qi + 1] = p[cj + 1];
                            bf[qi + 2] = p[cj + 2];
                            bf[qi + 3] = (j < tl) ? ap[j] : 255;
                        }
                    }
                if (depth == 2)
                    for (var y = 0; y < h; y++) {
                        var s0 = y * bpl, t0 = y * w;
                        for (var i = 0; i < w; i++) {
                            var qi = (t0 + i) << 2, j = ((data[s0 + (i >> 2)] >> (6 - ((i & 3) << 1))) & 3), cj = 3 * j;
                            bf[qi] = p[cj];
                            bf[qi + 1] = p[cj + 1];
                            bf[qi + 2] = p[cj + 2];
                            bf[qi + 3] = (j < tl) ? ap[j] : 255;
                        }
                    }
                if (depth == 4)
                    for (var y = 0; y < h; y++) {
                        var s0 = y * bpl, t0 = y * w;
                        for (var i = 0; i < w; i++) {
                            var qi = (t0 + i) << 2, j = ((data[s0 + (i >> 1)] >> (4 - ((i & 1) << 2))) & 15), cj = 3 * j;
                            bf[qi] = p[cj];
                            bf[qi + 1] = p[cj + 1];
                            bf[qi + 2] = p[cj + 2];
                            bf[qi + 3] = (j < tl) ? ap[j] : 255;
                        }
                    }
                if (depth == 8)
                    for (var i = 0; i < area; i++) {
                        var qi = i << 2, j = data[i], cj = 3 * j;
                        bf[qi] = p[cj];
                        bf[qi + 1] = p[cj + 1];
                        bf[qi + 2] = p[cj + 2];
                        bf[qi + 3] = (j < tl) ? ap[j] : 255;
                    }
            }
            else if (ctype == 4) { // gray + alpha
                if (depth == 8)
                    for (var i = 0; i < area; i++) {
                        var qi = i << 2, di = i << 1, gr = data[di];
                        bf[qi] = gr;
                        bf[qi + 1] = gr;
                        bf[qi + 2] = gr;
                        bf[qi + 3] = data[di + 1];
                    }
                if (depth == 16)
                    for (var i = 0; i < area; i++) {
                        var qi = i << 2, di = i << 2, gr = data[di];
                        bf[qi] = gr;
                        bf[qi + 1] = gr;
                        bf[qi + 2] = gr;
                        bf[qi + 3] = data[di + 2];
                    }
            }
            else if (ctype == 0) { // gray
                var tr = out.tabs["tRNS"] ? out.tabs["tRNS"] : -1;
                if (depth == 1)
                    for (var i = 0; i < area; i++) {
                        var gr = 255 * ((data[i >> 3] >> (7 - ((i & 7)))) & 1), al = (gr == tr * 255) ? 0 : 255;
                        bf32[i] = (al << 24) | (gr << 16) | (gr << 8) | gr;
                    }
                if (depth == 2)
                    for (var i = 0; i < area; i++) {
                        var gr = 85 * ((data[i >> 2] >> (6 - ((i & 3) << 1))) & 3), al = (gr == tr * 85) ? 0 : 255;
                        bf32[i] = (al << 24) | (gr << 16) | (gr << 8) | gr;
                    }
                if (depth == 4)
                    for (var i = 0; i < area; i++) {
                        var gr = 17 * ((data[i >> 1] >> (4 - ((i & 1) << 2))) & 15), al = (gr == tr * 17) ? 0 : 255;
                        bf32[i] = (al << 24) | (gr << 16) | (gr << 8) | gr;
                    }
                if (depth == 8)
                    for (var i = 0; i < area; i++) {
                        var gr = data[i], al = (gr == tr) ? 0 : 255;
                        bf32[i] = (al << 24) | (gr << 16) | (gr << 8) | gr;
                    }
                if (depth == 16)
                    for (var i = 0; i < area; i++) {
                        var gr = data[i << 1], al = (rs(data, i << 1) == tr) ? 0 : 255;
                        bf32[i] = (al << 24) | (gr << 16) | (gr << 8) | gr;
                    }
            }
            return bf;
        };
        UPNG.decode = function (buff) {
            var data = new Uint8Array(buff), offset = 8, bin = UPNG._bin, rUs = bin.readUshort, rUi = bin.readUint;
            var out = {
                tabs: {},
                frames: []
            };
            var dd = new Uint8Array(data.length), doff = 0; // put all IDAT data into it
            var fd, foff = 0; // frames
            var mgck = [0x89, 0x50, 0x4e, 0x47, 0x0d, 0x0a, 0x1a, 0x0a];
            for (var i = 0; i < 8; i++)
                if (data[i] != mgck[i])
                    throw "The input is not a PNG file!";
            while (offset < data.length) {
                var len = bin.readUint(data, offset);
                offset += 4;
                var type = bin.readASCII(data, offset, 4);
                offset += 4;
                //console.log(type,len);
                if (type == "IHDR") {
                    UPNG.decode._IHDR(data, offset, out);
                }
                else if (type == "IDAT") {
                    for (var i = 0; i < len; i++)
                        dd[doff + i] = data[offset + i];
                    doff += len;
                }
                else if (type == "acTL") {
                    out.tabs[type] = {
                        num_frames: rUi(data, offset),
                        num_plays: rUi(data, offset + 4)
                    };
                    fd = new Uint8Array(data.length);
                }
                else if (type == "fcTL") {
                    if (foff != 0) {
                        var fr = out.frames[out.frames.length - 1];
                        fr.data = UPNG.decode._decompress(out, fd.slice(0, foff), fr.rect.width, fr.rect.height);
                        foff = 0;
                    }
                    var rct = {
                        x: rUi(data, offset + 12),
                        y: rUi(data, offset + 16),
                        width: rUi(data, offset + 4),
                        height: rUi(data, offset + 8)
                    };
                    var del = rUs(data, offset + 22);
                    del = rUs(data, offset + 20) / (del == 0 ? 100 : del);
                    var frm = {
                        rect: rct,
                        delay: Math.round(del * 1000),
                        dispose: data[offset + 24],
                        blend: data[offset + 25]
                    };
                    //console.log(frm);
                    out.frames.push(frm);
                }
                else if (type == "fdAT") {
                    for (var i = 0; i < len - 4; i++)
                        fd[foff + i] = data[offset + i + 4];
                    foff += len - 4;
                }
                else if (type == "pHYs") {
                    out.tabs[type] = [bin.readUint(data, offset), bin.readUint(data, offset + 4), data[offset + 8]];
                }
                else if (type == "cHRM") {
                    out.tabs[type] = [];
                    for (var i = 0; i < 8; i++)
                        out.tabs[type].push(bin.readUint(data, offset + i * 4));
                }
                else if (type == "tEXt") {
                    if (out.tabs[type] == null)
                        out.tabs[type] = {};
                    var nz = bin.nextZero(data, offset);
                    var keyw = bin.readASCII(data, offset, nz - offset);
                    var text = bin.readASCII(data, nz + 1, offset + len - nz - 1);
                    out.tabs[type][keyw] = text;
                }
                else if (type == "iTXt") {
                    if (out.tabs[type] == null)
                        out.tabs[type] = {};
                    var nz = 0, off = offset;
                    nz = bin.nextZero(data, off);
                    var keyw = bin.readASCII(data, off, nz - off);
                    off = nz + 1;
                    off += 2;
                    nz = bin.nextZero(data, off);
                    var ltag = bin.readASCII(data, off, nz - off);
                    off = nz + 1;
                    nz = bin.nextZero(data, off);
                    var tkeyw = bin.readUTF8(data, off, nz - off);
                    off = nz + 1;
                    var text = bin.readUTF8(data, off, len - (off - offset));
                    out.tabs[type][keyw] = text;
                }
                else if (type == "PLTE") {
                    out.tabs[type] = bin.readBytes(data, offset, len);
                }
                else if (type == "hIST") {
                    var pl = out.tabs["PLTE"].length / 3;
                    out.tabs[type] = [];
                    for (var i = 0; i < pl; i++)
                        out.tabs[type].push(rUs(data, offset + i * 2));
                }
                else if (type == "tRNS") {
                    if (out.ctype == 3)
                        out.tabs[type] = bin.readBytes(data, offset, len);
                    else if (out.ctype == 0)
                        out.tabs[type] = rUs(data, offset);
                    else if (out.ctype == 2)
                        out.tabs[type] = [rUs(data, offset), rUs(data, offset + 2), rUs(data, offset + 4)];
                    //else console.log("tRNS for unsupported color type",out.ctype, len);
                }
                else if (type == "gAMA")
                    out.tabs[type] = bin.readUint(data, offset) / 100000;
                else if (type == "sRGB")
                    out.tabs[type] = data[offset];
                else if (type == "bKGD") {
                    if (out.ctype == 0 || out.ctype == 4)
                        out.tabs[type] = [rUs(data, offset)];
                    else if (out.ctype == 2 || out.ctype == 6)
                        out.tabs[type] = [rUs(data, offset), rUs(data, offset + 2), rUs(data, offset + 4)];
                    else if (out.ctype == 3)
                        out.tabs[type] = data[offset];
                }
                else if (type == "IEND") {
                    break;
                }
                offset += len;
                var crc = bin.readUint(data, offset);
                offset += 4;
            }
            if (foff != 0) {
                var fr = out.frames[out.frames.length - 1];
                fr.data = UPNG.decode._decompress(out, fd.slice(0, foff), fr.rect.width, fr.rect.height);
                foff = 0;
            }
            out.data = UPNG.decode._decompress(out, dd, out.width, out.height);
            delete out.compress;
            delete out.interlace;
            delete out.filter;
            return out;
        };
        UPNG.decode._decompress = function (out, dd, w, h) {
            if (out.compress == 0)
                dd = UPNG.decode._inflate(dd);
            if (out.interlace == 0)
                dd = UPNG.decode._filterZero(dd, out, 0, w, h);
            else if (out.interlace == 1)
                dd = UPNG.decode._readInterlace(dd, out);
            return dd;
        };
        UPNG.decode._inflate = function (data) {
            return pako["inflate"](data);
        };
        UPNG.decode._readInterlace = function (data, out) {
            var w = out.width, h = out.height;
            var bpp = UPNG.decode._getBPP(out), cbpp = bpp >> 3, bpl = Math.ceil(w * bpp / 8);
            var img = new Uint8Array(h * bpl);
            var di = 0;
            var starting_row = [0, 0, 4, 0, 2, 0, 1];
            var starting_col = [0, 4, 0, 2, 0, 1, 0];
            var row_increment = [8, 8, 8, 4, 4, 2, 2];
            var col_increment = [8, 8, 4, 4, 2, 2, 1];
            var pass = 0;
            while (pass < 7) {
                var ri = row_increment[pass], ci = col_increment[pass];
                var sw = 0, sh = 0;
                var cr = starting_row[pass];
                while (cr < h) {
                    cr += ri;
                    sh++;
                }
                var cc = starting_col[pass];
                while (cc < w) {
                    cc += ci;
                    sw++;
                }
                var bpll = Math.ceil(sw * bpp / 8);
                UPNG.decode._filterZero(data, out, di, sw, sh);
                var y = 0, row = starting_row[pass];
                while (row < h) {
                    var col = starting_col[pass];
                    var cdi = (di + y * bpll) << 3;
                    while (col < w) {
                        if (bpp == 1) {
                            var val = data[cdi >> 3];
                            val = (val >> (7 - (cdi & 7))) & 1;
                            img[row * bpl + (col >> 3)] |= (val << (7 - ((col & 3) << 0)));
                        }
                        if (bpp == 2) {
                            var val = data[cdi >> 3];
                            val = (val >> (6 - (cdi & 7))) & 3;
                            img[row * bpl + (col >> 2)] |= (val << (6 - ((col & 3) << 1)));
                        }
                        if (bpp == 4) {
                            var val = data[cdi >> 3];
                            val = (val >> (4 - (cdi & 7))) & 15;
                            img[row * bpl + (col >> 1)] |= (val << (4 - ((col & 1) << 2)));
                        }
                        if (bpp >= 8) {
                            var ii = row * bpl + col * cbpp;
                            for (var j = 0; j < cbpp; j++)
                                img[ii + j] = data[(cdi >> 3) + j];
                        }
                        cdi += bpp;
                        col += ci;
                    }
                    y++;
                    row += ri;
                }
                if (sw * sh != 0)
                    di += sh * (1 + bpll);
                pass = pass + 1;
            }
            return img;
        };
        UPNG.decode._getBPP = function (out) {
            var noc = [1, null, 3, 1, 2, null, 4][out.ctype];
            return noc * out.depth;
        };
        UPNG.decode._filterZero = function (data, out, off, w, h) {
            var bpp = UPNG.decode._getBPP(out), bpl = Math.ceil(w * bpp / 8), paeth = UPNG.decode._paeth;
            bpp = Math.ceil(bpp / 8);
            for (var y = 0; y < h; y++) {
                var i = off + y * bpl, di = i + y + 1;
                var type = data[di - 1];
                if (type == 0)
                    for (var x = 0; x < bpl; x++)
                        data[i + x] = data[di + x];
                else if (type == 1) {
                    for (var x = 0; x < bpp; x++)
                        data[i + x] = data[di + x];
                    for (var x = bpp; x < bpl; x++)
                        data[i + x] = (data[di + x] + data[i + x - bpp]) & 255;
                }
                else if (y == 0) {
                    for (var x = 0; x < bpp; x++)
                        data[i + x] = data[di + x];
                    if (type == 2)
                        for (var x = bpp; x < bpl; x++)
                            data[i + x] = (data[di + x]) & 255;
                    if (type == 3)
                        for (var x = bpp; x < bpl; x++)
                            data[i + x] = (data[di + x] + (data[i + x - bpp] >> 1)) & 255;
                    if (type == 4)
                        for (var x = bpp; x < bpl; x++)
                            data[i + x] = (data[di + x] + paeth(data[i + x - bpp], 0, 0)) & 255;
                }
                else {
                    if (type == 2) {
                        for (var x = 0; x < bpl; x++)
                            data[i + x] = (data[di + x] + data[i + x - bpl]) & 255;
                    }
                    if (type == 3) {
                        for (var x = 0; x < bpp; x++)
                            data[i + x] = (data[di + x] + (data[i + x - bpl] >> 1)) & 255;
                        for (var x = bpp; x < bpl; x++)
                            data[i + x] = (data[di + x] + ((data[i + x - bpl] + data[i + x - bpp]) >> 1)) & 255;
                    }
                    if (type == 4) {
                        for (var x = 0; x < bpp; x++)
                            data[i + x] = (data[di + x] + paeth(0, data[i + x - bpl], 0)) & 255;
                        for (var x = bpp; x < bpl; x++)
                            data[i + x] = (data[di + x] + paeth(data[i + x - bpp], data[i + x - bpl], data[i + x - bpp - bpl])) & 255;
                    }
                }
            }
            return data;
        };
        UPNG.decode._paeth = function (a, b, c) {
            var p = a + b - c, pa = Math.abs(p - a), pb = Math.abs(p - b), pc = Math.abs(p - c);
            if (pa <= pb && pa <= pc)
                return a;
            else if (pb <= pc)
                return b;
            return c;
        };
        UPNG.decode._IHDR = function (data, offset, out) {
            var bin = UPNG._bin;
            out.width = bin.readUint(data, offset);
            offset += 4;
            out.height = bin.readUint(data, offset);
            offset += 4;
            out.depth = data[offset];
            offset++;
            out.ctype = data[offset];
            offset++;
            out.compress = data[offset];
            offset++;
            out.filter = data[offset];
            offset++;
            out.interlace = data[offset];
            offset++;
        };
        UPNG._bin = {
            nextZero: function (data, p) {
                while (data[p] != 0)
                    p++;
                return p;
            },
            readUshort: function (buff, p) {
                return (buff[p] << 8) | buff[p + 1];
            },
            writeUshort: function (buff, p, n) {
                buff[p] = (n >> 8) & 255;
                buff[p + 1] = n & 255;
            },
            readUint: function (buff, p) {
                return (buff[p] * (256 * 256 * 256)) + ((buff[p + 1] << 16) | (buff[p + 2] << 8) | buff[p + 3]);
            },
            writeUint: function (buff, p, n) {
                buff[p] = (n >> 24) & 255;
                buff[p + 1] = (n >> 16) & 255;
                buff[p + 2] = (n >> 8) & 255;
                buff[p + 3] = n & 255;
            },
            readASCII: function (buff, p, l) {
                var s = "";
                for (var i = 0; i < l; i++)
                    s += String.fromCharCode(buff[p + i]);
                return s;
            },
            writeASCII: function (data, p, s) {
                for (var i = 0; i < s.length; i++)
                    data[p + i] = s.charCodeAt(i);
            },
            readBytes: function (buff, p, l) {
                var arr = [];
                for (var i = 0; i < l; i++)
                    arr.push(buff[p + i]);
                return arr;
            },
            pad: function (n) {
                return n.length < 2 ? "0" + n : n;
            },
            readUTF8: function (buff, p, l) {
                var s = "", ns;
                for (var i = 0; i < l; i++)
                    s += "%" + UPNG._bin.pad(buff[p + i].toString(16));
                try {
                    ns = decodeURIComponent(s);
                }
                catch (e) {
                    return UPNG._bin.readASCII(buff, p, l);
                }
                return ns;
            }
        };
        UPNG._copyTile = function (sb, sw, sh, tb, tw, th, xoff, yoff, mode) {
            var w = Math.min(sw, tw), h = Math.min(sh, th);
            var si = 0, ti = 0;
            for (var y = 0; y < h; y++)
                for (var x = 0; x < w; x++) {
                    if (xoff >= 0 && yoff >= 0) {
                        si = (y * sw + x) << 2;
                        ti = ((yoff + y) * tw + xoff + x) << 2;
                    }
                    else {
                        si = ((-yoff + y) * sw - xoff + x) << 2;
                        ti = (y * tw + x) << 2;
                    }
                    if (mode == 0) {
                        tb[ti] = sb[si];
                        tb[ti + 1] = sb[si + 1];
                        tb[ti + 2] = sb[si + 2];
                        tb[ti + 3] = sb[si + 3];
                    }
                    else if (mode == 1) {
                        var fa = sb[si + 3] * (1 / 255), fr = sb[si] * fa, fg = sb[si + 1] * fa, fb = sb[si + 2] * fa;
                        var ba = tb[ti + 3] * (1 / 255), br = tb[ti] * ba, bg = tb[ti + 1] * ba, bb = tb[ti + 2] * ba;
                        var ifa = 1 - fa, oa = fa + ba * ifa, ioa = (oa == 0 ? 0 : 1 / oa);
                        tb[ti + 3] = 255 * oa;
                        tb[ti + 0] = (fr + br * ifa) * ioa;
                        tb[ti + 1] = (fg + bg * ifa) * ioa;
                        tb[ti + 2] = (fb + bb * ifa) * ioa;
                    }
                    else if (mode == 2) { // copy only differences, otherwise zero
                        var fa = sb[si + 3], fr = sb[si], fg = sb[si + 1], fb = sb[si + 2];
                        var ba = tb[ti + 3], br = tb[ti], bg = tb[ti + 1], bb = tb[ti + 2];
                        if (fa == ba && fr == br && fg == bg && fb == bb) {
                            tb[ti] = 0;
                            tb[ti + 1] = 0;
                            tb[ti + 2] = 0;
                            tb[ti + 3] = 0;
                        }
                        else {
                            tb[ti] = fr;
                            tb[ti + 1] = fg;
                            tb[ti + 2] = fb;
                            tb[ti + 3] = fa;
                        }
                    }
                    else if (mode == 3) { // check if can be blended
                        var fa = sb[si + 3], fr = sb[si], fg = sb[si + 1], fb = sb[si + 2];
                        var ba = tb[ti + 3], br = tb[ti], bg = tb[ti + 1], bb = tb[ti + 2];
                        if (fa == ba && fr == br && fg == bg && fb == bb)
                            continue;
                        //if(fa!=255 && ba!=0) return false;
                        if (fa < 220 && ba > 20)
                            return false;
                    }
                }
            return true;
        };
        UPNG.encode = function (bufs, w, h, ps, dels, forbidPlte) {
            if (ps == null)
                ps = 0;
            if (forbidPlte == null)
                forbidPlte = false;
            var nimg = UPNG.encode.compress(bufs, w, h, ps, false, forbidPlte);
            UPNG.encode.compressPNG(nimg, -1);
            return UPNG.encode._main(nimg, w, h, dels);
        };
        UPNG.encodeLL = function (bufs, w, h, cc, ac, depth, dels) {
            var nimg = {
                ctype: 0 + (cc == 1 ? 0 : 2) + (ac == 0 ? 0 : 4),
                depth: depth,
                frames: []
            };
            var bipp = (cc + ac) * depth, bipl = bipp * w;
            for (var i = 0; i < bufs.length; i++)
                nimg.frames.push({
                    rect: {
                        x: 0,
                        y: 0,
                        width: w,
                        height: h
                    },
                    img: new Uint8Array(bufs[i]),
                    blend: 0,
                    dispose: 1,
                    bpp: Math.ceil(bipp / 8),
                    bpl: Math.ceil(bipl / 8)
                });
            UPNG.encode.compressPNG(nimg, 4);
            return UPNG.encode._main(nimg, w, h, dels);
        };
        UPNG.encode._main = function (nimg, w, h, dels) {
            var crc = UPNG.crc.crc, wUi = UPNG._bin.writeUint, wUs = UPNG._bin.writeUshort, wAs = UPNG._bin.writeASCII;
            var offset = 8, anim = nimg.frames.length > 1, pltAlpha = false;
            var leng = 8 + (16 + 5 + 4) + (9 + 4) + (anim ? 20 : 0);
            if (nimg.ctype == 3) {
                var dl = nimg.plte.length;
                for (var i = 0; i < dl; i++)
                    if ((nimg.plte[i] >>> 24) != 255)
                        pltAlpha = true;
                leng += (8 + dl * 3 + 4) + (pltAlpha ? (8 + dl * 1 + 4) : 0);
            }
            for (var j = 0; j < nimg.frames.length; j++) {
                var fr = nimg.frames[j];
                if (anim)
                    leng += 38;
                leng += fr.cimg.length + 12;
                if (j != 0)
                    leng += 4;
            }
            leng += 12;
            var data = new Uint8Array(leng);
            var wr = [0x89, 0x50, 0x4e, 0x47, 0x0d, 0x0a, 0x1a, 0x0a];
            for (var i = 0; i < 8; i++)
                data[i] = wr[i];
            wUi(data, offset, 13);
            offset += 4;
            wAs(data, offset, "IHDR");
            offset += 4;
            wUi(data, offset, w);
            offset += 4;
            wUi(data, offset, h);
            offset += 4;
            data[offset] = nimg.depth;
            offset++; // depth
            data[offset] = nimg.ctype;
            offset++; // ctype
            data[offset] = 0;
            offset++; // compress
            data[offset] = 0;
            offset++; // filter
            data[offset] = 0;
            offset++; // interlace
            wUi(data, offset, crc(data, offset - 17, 17));
            offset += 4; // crc
            // 9 bytes to say, that it is sRGB
            wUi(data, offset, 1);
            offset += 4;
            wAs(data, offset, "sRGB");
            offset += 4;
            data[offset] = 1;
            offset++;
            wUi(data, offset, crc(data, offset - 5, 5));
            offset += 4; // crc
            if (anim) {
                wUi(data, offset, 8);
                offset += 4;
                wAs(data, offset, "acTL");
                offset += 4;
                wUi(data, offset, nimg.frames.length);
                offset += 4;
                wUi(data, offset, 0);
                offset += 4;
                wUi(data, offset, crc(data, offset - 12, 12));
                offset += 4; // crc
            }
            if (nimg.ctype == 3) {
                var dl = nimg.plte.length;
                wUi(data, offset, dl * 3);
                offset += 4;
                wAs(data, offset, "PLTE");
                offset += 4;
                for (var i = 0; i < dl; i++) {
                    var ti = i * 3, c = nimg.plte[i], r = (c) & 255, g = (c >>> 8) & 255, b = (c >>> 16) & 255;
                    data[offset + ti + 0] = r;
                    data[offset + ti + 1] = g;
                    data[offset + ti + 2] = b;
                }
                offset += dl * 3;
                wUi(data, offset, crc(data, offset - dl * 3 - 4, dl * 3 + 4));
                offset += 4; // crc
                if (pltAlpha) {
                    wUi(data, offset, dl);
                    offset += 4;
                    wAs(data, offset, "tRNS");
                    offset += 4;
                    for (var i = 0; i < dl; i++)
                        data[offset + i] = (nimg.plte[i] >>> 24) & 255;
                    offset += dl;
                    wUi(data, offset, crc(data, offset - dl - 4, dl + 4));
                    offset += 4; // crc
                }
            }
            var fi = 0;
            for (var j = 0; j < nimg.frames.length; j++) {
                var fr = nimg.frames[j];
                if (anim) {
                    wUi(data, offset, 26);
                    offset += 4;
                    wAs(data, offset, "fcTL");
                    offset += 4;
                    wUi(data, offset, fi++);
                    offset += 4;
                    wUi(data, offset, fr.rect.width);
                    offset += 4;
                    wUi(data, offset, fr.rect.height);
                    offset += 4;
                    wUi(data, offset, fr.rect.x);
                    offset += 4;
                    wUi(data, offset, fr.rect.y);
                    offset += 4;
                    wUs(data, offset, dels[j]);
                    offset += 2;
                    wUs(data, offset, 1000);
                    offset += 2;
                    data[offset] = fr.dispose;
                    offset++; // dispose
                    data[offset] = fr.blend;
                    offset++; // blend
                    wUi(data, offset, crc(data, offset - 30, 30));
                    offset += 4; // crc
                }
                var imgd = fr.cimg, dl = imgd.length;
                wUi(data, offset, dl + (j == 0 ? 0 : 4));
                offset += 4;
                var ioff = offset;
                wAs(data, offset, (j == 0) ? "IDAT" : "fdAT");
                offset += 4;
                if (j != 0) {
                    wUi(data, offset, fi++);
                    offset += 4;
                }
                for (var i = 0; i < dl; i++)
                    data[offset + i] = imgd[i];
                offset += dl;
                wUi(data, offset, crc(data, ioff, offset - ioff));
                offset += 4; // crc
            }
            wUi(data, offset, 0);
            offset += 4;
            wAs(data, offset, "IEND");
            offset += 4;
            wUi(data, offset, crc(data, offset - 4, 4));
            offset += 4; // crc
            return data.buffer;
        };
        UPNG.encode.compressPNG = function (out, filter) {
            for (var i = 0; i < out.frames.length; i++) {
                var frm = out.frames[i], nw = frm.rect.width, nh = frm.rect.height;
                var fdata = new Uint8Array(nh * frm.bpl + nh);
                frm.cimg = UPNG.encode._filterZero(frm.img, nh, frm.bpp, frm.bpl, fdata, filter);
            }
        };
        UPNG.encode.compress = function (bufs, w, h, ps, forGIF, forbidPlte) {
            //var time = Date.now();
            if (forbidPlte == null)
                forbidPlte = false;
            var ctype = 6, depth = 8, alphaAnd = 255;
            for (var j = 0; j < bufs.length; j++) { // when not quantized, other frames can contain colors, that are not in an initial frame
                var img = new Uint8Array(bufs[j]), ilen = img.length;
                for (var i = 0; i < ilen; i += 4)
                    alphaAnd &= img[i + 3];
            }
            var gotAlpha = (alphaAnd != 255);
            //console.log("alpha check", Date.now()-time);  time = Date.now();
            var brute = gotAlpha && forGIF; // brute : frames can only be copied, not "blended"
            var frms = UPNG.encode.framize(bufs, w, h, forGIF, brute);
            //console.log("framize", Date.now()-time);  time = Date.now();
            var cmap = {}, plte = [], inds = [];
            if (ps != 0) {
                var nbufs = [];
                for (var i = 0; i < frms.length; i++)
                    nbufs.push(frms[i].img.buffer);
                var abuf = UPNG.encode.concatRGBA(nbufs, forGIF), qres = UPNG.quantize(abuf, ps);
                var cof = 0, bb = new Uint8Array(qres.abuf);
                for (var i = 0; i < frms.length; i++) {
                    var ti = frms[i].img, bln = ti.length;
                    inds.push(new Uint8Array(qres.inds.buffer, cof >> 2, bln >> 2));
                    for (var j = 0; j < bln; j += 4) {
                        ti[j] = bb[cof + j];
                        ti[j + 1] = bb[cof + j + 1];
                        ti[j + 2] = bb[cof + j + 2];
                        ti[j + 3] = bb[cof + j + 3];
                    }
                    cof += bln;
                }
                for (var i = 0; i < qres.plte.length; i++)
                    plte.push(qres.plte[i].est.rgba);
                //console.log("quantize", Date.now()-time);  time = Date.now();
            }
            else {
                // what if ps==0, but there are <=256 colors?  we still need to detect, if the palette could be used
                for (var j = 0; j < frms.length; j++) { // when not quantized, other frames can contain colors, that are not in an initial frame
                    var frm = frms[j], img32 = new Uint32Array(frm.img.buffer), nw = frm.rect.width, ilen = img32.length;
                    var ind = new Uint8Array(ilen);
                    inds.push(ind);
                    for (var i = 0; i < ilen; i++) {
                        var c = img32[i];
                        if (i != 0 && c == img32[i - 1])
                            ind[i] = ind[i - 1];
                        else if (i > nw && c == img32[i - nw])
                            ind[i] = ind[i - nw];
                        else {
                            var cmc = cmap[c];
                            if (cmc == null) {
                                cmap[c] = cmc = plte.length;
                                plte.push(c);
                                if (plte.length >= 300)
                                    break;
                            }
                            ind[i] = cmc;
                        }
                    }
                }
                //console.log("make palette", Date.now()-time);  time = Date.now();
            }
            var cc = plte.length; //console.log("colors:",cc);
            if (cc <= 256 && forbidPlte == false) {
                if (cc <= 2)
                    depth = 1;
                else if (cc <= 4)
                    depth = 2;
                else if (cc <= 16)
                    depth = 4;
                else
                    depth = 8;
                if (forGIF)
                    depth = 8;
            }
            for (var j = 0; j < frms.length; j++) {
                var frm = frms[j], nx = frm.rect.x, ny = frm.rect.y, nw = frm.rect.width, nh = frm.rect.height;
                var cimg = frm.img, cimg32 = new Uint32Array(cimg.buffer);
                var bpl = 4 * nw, bpp = 4;
                if (cc <= 256 && forbidPlte == false) {
                    bpl = Math.ceil(depth * nw / 8);
                    var nimg = new Uint8Array(bpl * nh);
                    var inj = inds[j];
                    for (var y = 0; y < nh; y++) {
                        var i = y * bpl, ii = y * nw;
                        if (depth == 8)
                            for (var x = 0; x < nw; x++)
                                nimg[i + (x)] = (inj[ii + x]);
                        else if (depth == 4)
                            for (var x = 0; x < nw; x++)
                                nimg[i + (x >> 1)] |= (inj[ii + x] << (4 - (x & 1) * 4));
                        else if (depth == 2)
                            for (var x = 0; x < nw; x++)
                                nimg[i + (x >> 2)] |= (inj[ii + x] << (6 - (x & 3) * 2));
                        else if (depth == 1)
                            for (var x = 0; x < nw; x++)
                                nimg[i + (x >> 3)] |= (inj[ii + x] << (7 - (x & 7) * 1));
                    }
                    cimg = nimg;
                    ctype = 3;
                    bpp = 1;
                }
                else if (gotAlpha == false && frms.length == 1) { // some next "reduced" frames may contain alpha for blending
                    var nimg = new Uint8Array(nw * nh * 3), area = nw * nh;
                    for (var i = 0; i < area; i++) {
                        var ti = i * 3, qi = i * 4;
                        nimg[ti] = cimg[qi];
                        nimg[ti + 1] = cimg[qi + 1];
                        nimg[ti + 2] = cimg[qi + 2];
                    }
                    cimg = nimg;
                    ctype = 2;
                    bpp = 3;
                    bpl = 3 * nw;
                }
                frm.img = cimg;
                frm.bpl = bpl;
                frm.bpp = bpp;
            }
            //console.log("colors => palette indices", Date.now()-time);  time = Date.now();
            return {
                ctype: ctype,
                depth: depth,
                plte: plte,
                frames: frms
            };
        };
        UPNG.encode.framize = function (bufs, w, h, forGIF, brute) {
            var frms = [];
            for (var j = 0; j < bufs.length; j++) {
                var cimg = new Uint8Array(bufs[j]), cimg32 = new Uint32Array(cimg.buffer);
                var nx = 0, ny = 0, nw = w, nh = h, blend = 0;
                if (j != 0 && !brute) {
                    var tlim = (forGIF || j == 1 || frms[frms.length - 2].dispose == 2) ? 1 : 2, tstp = 0, tarea = 1e9;
                    for (var it = 0; it < tlim; it++) {
                        var pimg = new Uint8Array(bufs[j - 1 - it]), p32 = new Uint32Array(bufs[j - 1 - it]);
                        var mix = w, miy = h, max = -1, may = -1;
                        for (var y = 0; y < h; y++)
                            for (var x = 0; x < w; x++) {
                                var i = y * w + x;
                                if (cimg32[i] != p32[i]) {
                                    if (x < mix)
                                        mix = x;
                                    if (x > max)
                                        max = x;
                                    if (y < miy)
                                        miy = y;
                                    if (y > may)
                                        may = y;
                                }
                            }
                        var sarea = (max == -1) ? 1 : (max - mix + 1) * (may - miy + 1);
                        if (sarea < tarea) {
                            tarea = sarea;
                            tstp = it;
                            if (max == -1) {
                                nx = ny = 0;
                                nw = nh = 1;
                            }
                            else {
                                nx = mix;
                                ny = miy;
                                nw = max - mix + 1;
                                nh = may - miy + 1;
                            }
                        }
                    }
                    var pimg = new Uint8Array(bufs[j - 1 - tstp]);
                    if (tstp == 1)
                        frms[frms.length - 1].dispose = 2;
                    var nimg = new Uint8Array(nw * nh * 4);
                    UPNG._copyTile(pimg, w, h, nimg, nw, nh, -nx, -ny, 0);
                    if (UPNG._copyTile(cimg, w, h, nimg, nw, nh, -nx, -ny, 3)) {
                        UPNG._copyTile(cimg, w, h, nimg, nw, nh, -nx, -ny, 2);
                        blend = 1;
                    }
                    else {
                        UPNG._copyTile(cimg, w, h, nimg, nw, nh, -nx, -ny, 0);
                        blend = 0;
                    }
                    cimg = nimg;
                }
                else
                    cimg = cimg.slice(0); // img may be rewrited further ... don't rewrite input
                frms.push({
                    rect: {
                        x: nx,
                        y: ny,
                        width: nw,
                        height: nh
                    },
                    img: cimg,
                    blend: blend,
                    dispose: brute ? 1 : 0
                });
            }
            return frms;
        };
        UPNG.encode._filterZero = function (img, h, bpp, bpl, data, filter) {
            if (filter != -1) {
                for (var y = 0; y < h; y++)
                    UPNG.encode._filterLine(data, img, y, bpl, bpp, filter);
                return pako["deflate"](data);
            }
            var fls = [];
            for (var t = 0; t < 5; t++) {
                if (h * bpl > 500000 && (t == 2 || t == 3 || t == 4))
                    continue;
                for (var y = 0; y < h; y++)
                    UPNG.encode._filterLine(data, img, y, bpl, bpp, t);
                fls.push(pako["deflate"](data));
                if (bpp == 1)
                    break;
            }
            var ti, tsize = 1e9;
            for (var i = 0; i < fls.length; i++)
                if (fls[i].length < tsize) {
                    ti = i;
                    tsize = fls[i].length;
                }
            return fls[ti];
        };
        UPNG.encode._filterLine = function (data, img, y, bpl, bpp, type) {
            var i = y * bpl, di = i + y, paeth = UPNG.decode._paeth;
            data[di] = type;
            di++;
            if (type == 0)
                for (var x = 0; x < bpl; x++)
                    data[di + x] = img[i + x];
            else if (type == 1) {
                for (var x = 0; x < bpp; x++)
                    data[di + x] = img[i + x];
                for (var x = bpp; x < bpl; x++)
                    data[di + x] = (img[i + x] - img[i + x - bpp] + 256) & 255;
            }
            else if (y == 0) {
                for (var x = 0; x < bpp; x++)
                    data[di + x] = img[i + x];
                if (type == 2)
                    for (var x = bpp; x < bpl; x++)
                        data[di + x] = img[i + x];
                if (type == 3)
                    for (var x = bpp; x < bpl; x++)
                        data[di + x] = (img[i + x] - (img[i + x - bpp] >> 1) + 256) & 255;
                if (type == 4)
                    for (var x = bpp; x < bpl; x++)
                        data[di + x] = (img[i + x] - paeth(img[i + x - bpp], 0, 0) + 256) & 255;
            }
            else {
                if (type == 2) {
                    for (var x = 0; x < bpl; x++)
                        data[di + x] = (img[i + x] + 256 - img[i + x - bpl]) & 255;
                }
                if (type == 3) {
                    for (var x = 0; x < bpp; x++)
                        data[di + x] = (img[i + x] + 256 - (img[i + x - bpl] >> 1)) & 255;
                    for (var x = bpp; x < bpl; x++)
                        data[di + x] = (img[i + x] + 256 - ((img[i + x - bpl] + img[i + x - bpp]) >> 1)) & 255;
                }
                if (type == 4) {
                    for (var x = 0; x < bpp; x++)
                        data[di + x] = (img[i + x] + 256 - paeth(0, img[i + x - bpl], 0)) & 255;
                    for (var x = bpp; x < bpl; x++)
                        data[di + x] = (img[i + x] + 256 - paeth(img[i + x - bpp], img[i + x - bpl], img[i + x - bpp - bpl])) & 255;
                }
            }
        };
        UPNG.crc = {
            table: (function () {
                var tab = new Uint32Array(256);
                for (var n = 0; n < 256; n++) {
                    var c = n;
                    for (var k = 0; k < 8; k++) {
                        if (c & 1)
                            c = 0xedb88320 ^ (c >>> 1);
                        else
                            c = c >>> 1;
                    }
                    tab[n] = c;
                }
                return tab;
            })(),
            update: function (c, buf, off, len) {
                for (var i = 0; i < len; i++)
                    c = UPNG.crc.table[(c ^ buf[off + i]) & 0xff] ^ (c >>> 8);
                return c;
            },
            crc: function (b, o, l) {
                return UPNG.crc.update(0xffffffff, b, o, l) ^ 0xffffffff;
            }
        };
        UPNG.quantize = function (abuf, ps) {
            var oimg = new Uint8Array(abuf), nimg = oimg.slice(0), nimg32 = new Uint32Array(nimg.buffer);
            var KD = UPNG.quantize.getKDtree(nimg, ps);
            var root = KD[0], leafs = KD[1];
            var planeDst = UPNG.quantize.planeDst;
            var sb = oimg, tb = nimg32, len = sb.length;
            var inds = new Uint8Array(oimg.length >> 2);
            for (var i = 0; i < len; i += 4) {
                var r = sb[i] * (1 / 255), g = sb[i + 1] * (1 / 255), b = sb[i + 2] * (1 / 255), a = sb[i + 3] * (1 / 255);
                //  exact, but too slow :(
                var nd = UPNG.quantize.getNearest(root, r, g, b, a);
                //var nd = root;
                //while(nd.left) nd = (planeDst(nd.est,r,g,b,a)<=0) ? nd.left : nd.right;
                inds[i >> 2] = nd.ind;
                tb[i >> 2] = nd.est.rgba;
            }
            return {
                abuf: nimg.buffer,
                inds: inds,
                plte: leafs
            };
        };
        UPNG.quantize.getKDtree = function (nimg, ps, err) {
            if (err == null)
                err = 0.0001;
            var nimg32 = new Uint32Array(nimg.buffer);
            var root = {
                i0: 0,
                i1: nimg.length,
                bst: null,
                est: null,
                tdst: 0,
                left: null,
                right: null
            }; // basic statistic, extra statistic
            root.bst = UPNG.quantize.stats(nimg, root.i0, root.i1);
            root.est = UPNG.quantize.estats(root.bst);
            var leafs = [root];
            while (leafs.length < ps) {
                var maxL = 0, mi = 0;
                for (var i = 0; i < leafs.length; i++)
                    if (leafs[i].est.L > maxL) {
                        maxL = leafs[i].est.L;
                        mi = i;
                    }
                if (maxL < err)
                    break;
                var node = leafs[mi];
                var s0 = UPNG.quantize.splitPixels(nimg, nimg32, node.i0, node.i1, node.est.e, node.est.eMq255);
                var s0wrong = (node.i0 >= s0 || node.i1 <= s0);
                //console.log(maxL, leafs.length, mi);
                if (s0wrong) {
                    node.est.L = 0;
                    continue;
                }
                var ln = {
                    i0: node.i0,
                    i1: s0,
                    bst: null,
                    est: null,
                    tdst: 0,
                    left: null,
                    right: null
                };
                ln.bst = UPNG.quantize.stats(nimg, ln.i0, ln.i1);
                ln.est = UPNG.quantize.estats(ln.bst);
                var rn = {
                    i0: s0,
                    i1: node.i1,
                    bst: null,
                    est: null,
                    tdst: 0,
                    left: null,
                    right: null
                };
                rn.bst = {
                    R: [],
                    m: [],
                    N: node.bst.N - ln.bst.N
                };
                for (var i = 0; i < 16; i++)
                    rn.bst.R[i] = node.bst.R[i] - ln.bst.R[i];
                for (var i = 0; i < 4; i++)
                    rn.bst.m[i] = node.bst.m[i] - ln.bst.m[i];
                rn.est = UPNG.quantize.estats(rn.bst);
                node.left = ln;
                node.right = rn;
                leafs[mi] = ln;
                leafs.push(rn);
            }
            leafs.sort(function (a, b) {
                return b.bst.N - a.bst.N;
            });
            for (var i = 0; i < leafs.length; i++)
                leafs[i].ind = i;
            return [root, leafs];
        };
        UPNG.quantize.getNearest = function (nd, r, g, b, a) {
            if (nd.left == null) {
                nd.tdst = UPNG.quantize.dist(nd.est.q, r, g, b, a);
                return nd;
            }
            var planeDst = UPNG.quantize.planeDst(nd.est, r, g, b, a);
            var node0 = nd.left, node1 = nd.right;
            if (planeDst > 0) {
                node0 = nd.right;
                node1 = nd.left;
            }
            var ln = UPNG.quantize.getNearest(node0, r, g, b, a);
            if (ln.tdst <= planeDst * planeDst)
                return ln;
            var rn = UPNG.quantize.getNearest(node1, r, g, b, a);
            return rn.tdst < ln.tdst ? rn : ln;
        };
        UPNG.quantize.planeDst = function (est, r, g, b, a) {
            var e = est.e;
            return e[0] * r + e[1] * g + e[2] * b + e[3] * a - est.eMq;
        };
        UPNG.quantize.dist = function (q, r, g, b, a) {
            var d0 = r - q[0], d1 = g - q[1], d2 = b - q[2], d3 = a - q[3];
            return d0 * d0 + d1 * d1 + d2 * d2 + d3 * d3;
        };
        UPNG.quantize.splitPixels = function (nimg, nimg32, i0, i1, e, eMq) {
            var vecDot = UPNG.quantize.vecDot;
            i1 -= 4;
            while (i0 < i1) {
                while (vecDot(nimg, i0, e) <= eMq)
                    i0 += 4;
                while (vecDot(nimg, i1, e) > eMq)
                    i1 -= 4;
                if (i0 >= i1)
                    break;
                var t = nimg32[i0 >> 2];
                nimg32[i0 >> 2] = nimg32[i1 >> 2];
                nimg32[i1 >> 2] = t;
                i0 += 4;
                i1 -= 4;
            }
            while (vecDot(nimg, i0, e) > eMq)
                i0 -= 4;
            return i0 + 4;
        };
        UPNG.quantize.vecDot = function (nimg, i, e) {
            return nimg[i] * e[0] + nimg[i + 1] * e[1] + nimg[i + 2] * e[2] + nimg[i + 3] * e[3];
        };
        UPNG.quantize.stats = function (nimg, i0, i1) {
            var R = [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0];
            var m = [0, 0, 0, 0];
            var N = (i1 - i0) >> 2;
            for (var i = i0; i < i1; i += 4) {
                var r = nimg[i] * (1 / 255), g = nimg[i + 1] * (1 / 255), b = nimg[i + 2] * (1 / 255), a = nimg[i + 3] * (1 / 255);
                //var r = nimg[i], g = nimg[i+1], b = nimg[i+2], a = nimg[i+3];
                m[0] += r;
                m[1] += g;
                m[2] += b;
                m[3] += a;
                R[0] += r * r;
                R[1] += r * g;
                R[2] += r * b;
                R[3] += r * a;
                R[5] += g * g;
                R[6] += g * b;
                R[7] += g * a;
                R[10] += b * b;
                R[11] += b * a;
                R[15] += a * a;
            }
            R[4] = R[1];
            R[8] = R[2];
            R[9] = R[6];
            R[12] = R[3];
            R[13] = R[7];
            R[14] = R[11];
            return {
                R: R,
                m: m,
                N: N
            };
        };
        UPNG.quantize.estats = function (stats) {
            var R = stats.R, m = stats.m, N = stats.N;
            // when all samples are equal, but N is large (millions), the Rj can be non-zero ( 0.0003.... - precission error)
            var m0 = m[0], m1 = m[1], m2 = m[2], m3 = m[3], iN = (N == 0 ? 0 : 1 / N);
            var Rj = [R[0] - m0 * m0 * iN, R[1] - m0 * m1 * iN, R[2] - m0 * m2 * iN, R[3] - m0 * m3 * iN, R[4] - m1 * m0 * iN, R[5] - m1 * m1 * iN, R[6] - m1 * m2 * iN, R[7] - m1 * m3 * iN, R[8] - m2 * m0 * iN, R[9] - m2 * m1 * iN, R[10] - m2 * m2 * iN, R[11] - m2 * m3 * iN, R[12] - m3 * m0 * iN, R[13] - m3 * m1 * iN, R[14] - m3 * m2 * iN, R[15] - m3 * m3 * iN];
            var A = Rj, M = UPNG.M4;
            var b = [0.5, 0.5, 0.5, 0.5], mi = 0, tmi = 0;
            if (N != 0)
                for (var i = 0; i < 10; i++) {
                    b = M.multVec(A, b);
                    tmi = Math.sqrt(M.dot(b, b));
                    b = M.sml(1 / tmi, b);
                    if (Math.abs(tmi - mi) < 1e-9)
                        break;
                    mi = tmi;
                }
            //b = [0,0,1,0];  mi=N;
            var q = [m0 * iN, m1 * iN, m2 * iN, m3 * iN];
            var eMq255 = M.dot(M.sml(255, q), b);
            return {
                Cov: Rj,
                q: q,
                e: b,
                L: mi,
                eMq255: eMq255,
                eMq: M.dot(b, q),
                rgba: (((Math.round(255 * q[3]) << 24) | (Math.round(255 * q[2]) << 16) | (Math.round(255 * q[1]) << 8) | (Math.round(255 * q[0]) << 0)) >>> 0)
            };
        };
        UPNG.M4 = {
            multVec: function (m, v) {
                return [m[0] * v[0] + m[1] * v[1] + m[2] * v[2] + m[3] * v[3], m[4] * v[0] + m[5] * v[1] + m[6] * v[2] + m[7] * v[3], m[8] * v[0] + m[9] * v[1] + m[10] * v[2] + m[11] * v[3], m[12] * v[0] + m[13] * v[1] + m[14] * v[2] + m[15] * v[3]];
            },
            dot: function (x, y) {
                return x[0] * y[0] + x[1] * y[1] + x[2] * y[2] + x[3] * y[3];
            },
            sml: function (a, y) {
                return [a * y[0], a * y[1], a * y[2], a * y[3]];
            }
        };
        UPNG.encode.concatRGBA = function (bufs, roundAlpha) {
            var tlen = 0;
            for (var i = 0; i < bufs.length; i++)
                tlen += bufs[i].byteLength;
            var nimg = new Uint8Array(tlen), noff = 0;
            for (var i = 0; i < bufs.length; i++) {
                var img = new Uint8Array(bufs[i]), il = img.length;
                for (var j = 0; j < il; j += 4) {
                    var r = img[j], g = img[j + 1], b = img[j + 2], a = img[j + 3];
                    if (roundAlpha)
                        a = (a & 128) == 0 ? 0 : 255;
                    if (a == 0)
                        r = g = b = 0;
                    nimg[noff + j] = r;
                    nimg[noff + j + 1] = g;
                    nimg[noff + j + 2] = b;
                    nimg[noff + j + 3] = a;
                }
                noff += il;
            }
            return nimg.buffer;
        };
    })(UPNG, pako_1);

    var Image = /** @class */ (function () {
        function Image(esource, resources) {
            var _ts = this;
            _ts.esource = esource;
            _ts.resources = resources;
            _ts.init();
        }
        Image.prototype.init = function () {
            var _ts = this, esource = _ts.esource, resources = _ts.resources;
            _ts.temp = {
                //loop:0,                                       // 保存当前需要播放的次数
                //tickerIsAdd:undefined                         // 保存轮循执行器是否添加
                events: {} // 用于存放事件
            };
            // 属性
            _ts.__attr = {
                autoPlay: true,
                loop: 0 // 默认无限次播放
            };
            // 方法
            _ts.__method = {
                play: _ts.play // 播放方法
            };
            // 状态
            _ts.__status = {
                status: 'init',
                frame: 0,
                loops: 0,
                time: 0
            };
            // 循环执行器
            _ts.ticker = new PIXI.Ticker();
            _ts.ticker.stop();
            // 精灵
            _ts.sprite = this.createSprite(esource, resources);
        };
        // 播放
        Image.prototype.play = function (loop, callback) {
            var _ts = this;
            // 没有纹理材质时抛出错误
            if (!_ts.textures.length) {
                throw new Error('没有可用的textures');
            }
            // 纹理材质只有一帧时不往下执行
            if (_ts.textures.length === 1) {
                return;
            }
            var status = _ts.__status, attr = _ts.__attr, time = 0;
            // 当状态是停止的时候，将播放次数清0
            if (status.status === 'stop') {
                status.loops = 0;
            }
            // 设置循环参数
            loop = typeof loop === 'number' ? loop : attr.loop;
            _ts.temp.loop = loop;
            attr.loop = loop;
            // 为轮循执行器添加一个操作
            if (!_ts.temp.tickerIsAdd) {
                _ts.ticker.add(function (deltaTime) {
                    var elapsed = PIXI.Ticker.shared.elapsedMS;
                    time += elapsed;
                    // 当帧停留时间已达到间隔帧率时播放下一帧
                    if (time > _ts.framesDelay[status.frame]) {
                        status.frame++;
                        // 修改状态为执行中
                        status.status = 'playing';
                        // 当一次播放完成，将播放帧归0，并记录播放次数
                        if (status.frame > _ts.textures.length - 1) {
                            status.frame = 0;
                            status.loops++;
                            // 当指定了有效的播放次数并且当前播放次数达到指定次数时，执行回调则停止播放
                            if (_ts.temp.loop > 0 && status.loops >= _ts.temp.loop) {
                                if (typeof callback === 'function') {
                                    callback(status);
                                }
                                // 修改状态为执行完成并停止
                                status.status = 'played';
                                _ts.runEvent('played', status);
                                _ts.stop();
                            }
                        }
                        // 修改精灵纹理材质与当前的帧率相匹配
                        _ts.sprite.texture = _ts.textures[status.frame];
                        time = 0;
                        _ts.runEvent('playing', status);
                    }
                });
                _ts.temp.tickerIsAdd = true;
            }
            // 让轮循执行器开始执行
            _ts.ticker.start();
        };
        // 暂停
        Image.prototype.pause = function () {
            var _ts = this, status = _ts.__status;
            _ts.ticker.stop();
            status.status = 'pause';
            _ts.runEvent('pause', status);
        };
        // 停止播放并跳至第一帧
        Image.prototype.stop = function () {
            var _ts = this, status = _ts.__status;
            _ts.ticker.stop();
            status.status = 'stop';
            _ts.runEvent('stop', status);
        };
        // 跳至指定的帧数
        Image.prototype.jumpToFrame = function (frameIndex) {
            var _ts = this, textures = _ts.textures;
            // 没有纹理材质时抛出错误
            if (!textures.length) {
                throw new Error('没有可用的textures');
            }
            var status = _ts.__status;
            frameIndex = frameIndex < 0 ? 0 : frameIndex > textures.length - 1 ? textures.length - 1 : frameIndex;
            if (typeof frameIndex === 'number') {
                _ts.sprite.texture = textures[frameIndex];
                status.frame = frameIndex;
            }
        };
        // 获取总播放时长
        Image.prototype.getDuration = function () {
            var _ts = this, framesDelay = _ts.framesDelay;
            // 没有帧时间时抛出错误
            if (!framesDelay.length) {
                throw new Error('未找到图片帧时间');
            }
            var time = 0;
            for (var i = 0, len = framesDelay.length; i < len; i++) {
                time += framesDelay[i];
            }
            return time;
        };
        // 获取总帧数
        Image.prototype.getFramesLength = function () {
            var _ts = this;
            // 没有纹理材质时抛出错误
            if (!_ts.textures.length) {
                throw new Error('没有可用的textures');
            }
            return _ts.textures.length;
        };
        // 事件
        Image.prototype.on = function (type, fun) {
            var _ts = this;
            switch (type) {
                case 'playing':
                case 'played':
                case 'pause':
                case 'stop':
                    _ts.temp.events[type] = fun;
                    break;
                default:
                    throw new Error('无效的事件');
                    break;
            }
        };
        Image.prototype.runEvent = function (type, status) {
            var temp = this.temp;
            if (typeof temp.events[type] === 'function') {
                temp.events[type](status);
            }
        };
        /**
         * 创建精灵
         * @param  {array:string}} imgSrc 图片资源路径
         * @param  {object} resources 已经加载的缓存资源
         * @return {object} 返回精灵
         */
        Image.prototype.createSprite = function (esource, resources) {
            var _ts = this;
            var Sprite = PIXI.Sprite, imgSrc = esource, exeName = $getExeName(imgSrc.toLocaleLowerCase());
            // 文件扩展名为gif或png则返回对应的名称，其它反返回other
            exeName = exeName === 'gif' || exeName === 'png' ? exeName : 'other';
            var funs = {
                'gif': function () {
                    var gifDecodeData = _ts.gifResourceToTextures(resources[imgSrc]);
                    _ts.textures = gifDecodeData.textures;
                    _ts.framesDelay = gifDecodeData.delayTimes;
                    _ts.play();
                    // 返回精灵并将纹理材质设置为第一帧图像
                    return new Sprite(_ts.textures[0]);
                },
                'png': function () {
                    var pngDecodeData = _ts.apngResourceToTextures(resources[imgSrc]);
                    _ts.textures = pngDecodeData.textures;
                    _ts.framesDelay = pngDecodeData.delayTimes;
                    _ts.play();
                    // 返回精灵并将纹理材质设置为第一帧图像
                    return new Sprite(_ts.textures[0]);
                },
                'other': function () {
                    _ts.textures = [resources[imgSrc].texture];
                    return new Sprite(resources[imgSrc].texture);
                }
            };
            return funs[exeName]();
        };
        /**
         * 将apng缓存资源转换为纹理材质
         * @param  {object} resource    缓存资源
         * @return {object} 返回一个对象，包括apng的每帧时长及解码出来材质
         */
        Image.prototype.apngResourceToTextures = function (resource) {
            var obj = {
                delayTimes: [],
                textures: []
            }, buf = new Uint8Array(resource.data), upng = UPNG.decode(buf), rgba = UPNG.toRGBA8(upng), pngWidth = upng.width, pngHeight = upng.height, pngFramesLen = upng.frames.length, spriteSheet, canvas, ctx, imageData;
            // 记录下每帧的时间
            upng.frames.forEach(function (item, index) {
                obj.delayTimes.push(item.delay);
            });
            for (var i = 0, len = rgba.length; i < len; i++) {
                var item = rgba[i], data = new Uint8ClampedArray(item);
                canvas = document.createElement('canvas');
                canvas.width = pngWidth;
                canvas.height = pngHeight;
                ctx = canvas.getContext('2d');
                spriteSheet = new PIXI.BaseTexture.from(canvas);
                imageData = ctx.createImageData(pngWidth, pngHeight);
                imageData.data.set(data);
                ctx.putImageData(imageData, 0, 0);
                obj.textures.push(new PIXI.Texture(spriteSheet, new PIXI.Rectangle(0, 0, pngWidth, pngHeight)));
            }
            // document.body.appendChild(canvas);
            return obj;
        };
        /**
         * 将gif缓存资源转换为纹理材质
         * @param  {object} resource    缓存资源
         * @return {object} 返回一个对象，包括apng的每帧时长及解码出来材质
         */
        Image.prototype.gifResourceToTextures = function (resource) {
            var obj = {
                delayTimes: [],
                textures: []
            }, buf = new Uint8Array(resource.data), gif = new GifReader(buf), gifWidth = gif.width, gifHeight = gif.height, gifFramesLen = gif.numFrames(), gifFrameInfo, spriteSheet, canvas, ctx, imageData;
            for (var i = 0; i < gifFramesLen; i++) {
                //得到每帧的信息并将帧延迟信息保存起来
                gifFrameInfo = gif.frameInfo(i);
                obj.delayTimes.push(gifFrameInfo.delay * 10);
                canvas = document.createElement('canvas');
                canvas.width = gifWidth;
                canvas.height = gifHeight;
                ctx = canvas.getContext('2d');
                //创建一块空白的ImageData对象
                imageData = ctx.createImageData(gifWidth, gifHeight);
                //将第一帧转换为RGBA值，将赋予到图像区
                gif.decodeAndBlitFrameRGBA(i, imageData.data);
                //将上面创建的图像数据放回到画面上
                ctx.putImageData(imageData, 0, 0);
                spriteSheet = new PIXI.BaseTexture.from(canvas);
                obj.textures.push(new PIXI.Texture(spriteSheet, new PIXI.Rectangle(0, 0, gifWidth, gifHeight)));
            }
            // document.body.appendChild(canvas);
            return obj;
        };
        return Image;
    }());

    return Image;

})));
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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